package joyfill2.editors.table.internal

import cinematic.mutableLiveOf
import joyfill2.Document
import joyfill2.collections.PageCollection
import joyfill2.editors.components.internal.AbstractComponentEditor
import joyfill2.editors.table.RowManager
import joyfill2.editors.table.Selection
import joyfill2.editors.table.TableEditor
import joyfill2.events.ChangeEvent
import joyfill2.events.ChangeLog
import joyfill2.table.Cell
import joyfill2.table.Row
import joyfill2.table.TableComponent
import joyfill2.tools.validation.RowValid
import joyfill2.tools.validation.RowValidity
import joyfill2.tools.validation.TableInvalid
import joyfill2.tools.validation.TableValid
import joyfill2.tools.validation.TableValidity
import joyfill2.tools.visibility.Visibility

internal abstract class AbstractTableEditor(
    override val component: TableComponent,
    document: Document,
    pages: PageCollection,
    onChange: ((ChangeEvent) -> Unit)?,
    identifier: String,
    private val initialHiddenMap: Map<String, Boolean>,
    fieldId: String,
) : AbstractComponentEditor(document, pages, component, {}, onChange, identifier, fieldId), TableEditor {

    private val changeLogBuilder by lazy { TableChangeLogBuilder(component.columns) }
    private val conditionEvaluator by lazy { TableConditionEvaluator(::getCell) }

    override fun changeHiddenTo(value: Boolean) {
        hidden = value
        state.value = state.value.copy(
            hidden = value,
            visibility = if (value) Visibility.Hidden else Visibility.Visible,
        )
    }

    override val state by lazy {
        mutableLiveOf(
            TableEditorStateImpl(
                validity = TableValid(component, emptyList()),
                rows = rows.all(),
                selected = emptyList(),
                form = null,
                hidden = hidden,
                visibility = if (hidden) Visibility.Hidden else Visibility.Visible,
            )
        )
    }

    protected fun onRowValidated(validity: RowValidity) {
        val filteredRows = state.value.validity.rows.filterNot { it.row.id == validity.row.id }
        val updatedRows = filteredRows + validity
        val (valid, invalid) = updatedRows.partition { it is RowValid }
        val isTableValid = !component.required || rows.all().isNotEmpty()
        state.value = state.value.copy(
            validity = when {
                invalid.isEmpty() && isTableValid -> TableValid(component, valid)
                else -> TableInvalid(component, valid, invalid, updatedRows)
            }
        )
    }

    abstract fun getCell(column: String): Cell?

    override fun resolveConditions() {
        val logic = component.tableLogic ?: return
        val initialHiddenValue = initialHiddenMap[component.id] ?: return
        val value =
            conditionEvaluator.evaluateConditions(logic, state.value.hidden, initialHiddenValue)
        hidden = value
        state.value = state.value.copy(
            hidden = value
        )
    }

    override fun validate(): TableValidity {
        rows.all().flatMap { it.all() }.forEach { it.validate() }
        return state.value.validity
    }

    override fun select(item: RowManager?): RowManager? {
        item ?: return null
        if (item in state.value.selected) return item

        val selectedItem = item.select()

        state.value = state.value.copy(
            selected = state.value.selected + item,
            rows = state.value.rows.map {
                if (it.id == selectedItem.id) selectedItem else it
            }
        )

        return selectedItem
    }

    override fun unselect(item: RowManager?): RowManager? {
        item ?: return null
        val unselectedRow = item.unselect()

        state.value = state.value.copy(
            selected = state.value.selected - item,
            rows = state.value.rows.map {
                if (it.id == unselectedRow.id) unselectedRow else it
            }
        )

        return unselectedRow
    }

    override fun selectAll() {
        state.value = state.value.copy(
            selected = state.value.rows,
            rows = state.value.rows.map { it.select() }
        )
    }

    override fun unselectAll() {
        state.value = state.value.copy(
            rows = state.value.rows.map { it.unselect() },
            selected = emptyList()
        )
    }

    override fun edit() {
        val selected = state.value.selected
        if (selected.isEmpty()) return close()
        if (selected.size > 1) {
            val newRowManager = rows.create(true)
            state.value = state.value.copy(form = newRowManager)
        } else {
            state.value = state.value.copy(form = selected.first())
        }
    }

    override fun submit() {
        val form = state.value.form ?: return close()
        state.value.selected.forEach(form::copyInto)
        close()
    }

    override fun close() {
        state.value = state.value.copy(
            form = null
        )
    }

    protected fun next(): RowManager? {
        val currentForm = state.value.form
        val nextIndex = state.value.rows.indexOfFirst { it.row.id == currentForm?.row?.id } + 1
        val nextForm = state.value.rows.getOrNull(nextIndex) ?: return null
        state.value = state.value.copy(form = nextForm)
        return nextForm
    }

    protected fun prev(): RowManager? {
        val currentForm = state.value.form
        val prevIndex = state.value.rows.indexOfFirst { it.row.id == currentForm?.row?.id } - 1
        val prevForm = state.value.rows.getOrNull(prevIndex) ?: return null
        state.value = state.value.copy(form = prevForm)
        return prevForm
    }

    override fun selection(): Selection = state.value.selection()

    protected fun buildRowCreateChangeValue(rowManager: RowManager, index: Int) =
        changeLogBuilder.buildRowCreateChangeValue(rowManager, index)

    protected fun buildRowMoveChangeValue(row: Row?, index: Int) =
        changeLogBuilder.buildRowMoveChangeValue(row, index)

    protected fun buildRowDeleteChangeValue(rowManager: RowManager) =
        changeLogBuilder.buildRowDeleteChangeValue(rowManager)

    protected fun buildRowUpdateChangeValue(row: Row, changeLog: ChangeLog) =
        changeLogBuilder.buildRowUpdateChangeValue(row, changeLog)
}