package joyfill2.editors.table.internal

import joyfill2.editors.table.RowManager
import joyfill2.editors.table.RowResult
import joyfill2.editors.table.Rows
import joyfill2.table.Row
import joyfill2.value.ListValue

internal abstract class AbstractRows<M : RowManager>(
    private val component: ListValue<Row>,
    private val rowOrder: MutableList<String>
) : Rows<M> {

    protected val cache by lazy { mutableMapOf<String, M>() }

    abstract override fun create(empty: Boolean): M

    override fun append(): M {
        val manager = create()
        component.value.add(manager.row)
        rowOrder.add(manager.row.id)
        cache[manager.row.id] = manager
        return manager
    }

    override fun find(id: String): M? =
        (cache[id] ?: component.value.find { it.id == id && !it.deleted && it.id in rowOrder }?.toManager())

    override fun all(): List<M> = component.value.map { it.toManager() }.filter { !it.row.deleted && it.row.id in rowOrder }

    override fun delete(keys: List<String>): List<M> {
        val rows = all()
        val removed = rows.filter { it.row.id in keys }
        val retained = rows - removed.toSet()

        cache.keys.removeAll { it in keys }
        component.value.clear()
        component.value.addAll(retained.map { it.row })

        rowOrder.clear()
        rowOrder.addAll(retained.map { it.row.id })
        return removed
    }

    abstract fun Row.toManager(): M

    override fun addAfter(id: String): M? {
        val index = component.value.indexOfFirst { it.id == id }
        if (index == -1) return null
        val manager = create()

        component.value.add(index + 1, manager.row)
        rowOrder.add(index + 1, manager.row.id)
        cache[manager.row.id] = manager
        return manager
    }

    private fun moveRow(id: String, indexOffset: Int): RowResult<Row> {
        val currentRows = component.value.toList()
        val currentRowOrder = rowOrder.toList()
        val rowIndex = rowOrder.indexOfFirst { it == id }
        if (rowIndex == -1) {
            return RowResult(null, -1)
        }

        val row = currentRows[rowIndex]

        val targetIndex = (rowIndex + indexOffset).coerceIn(0, currentRowOrder.lastIndex)

        if (rowIndex == targetIndex){
            return RowResult(row, rowIndex)
        }

        val newRows = currentRows.toMutableList().apply {
            removeAt(rowIndex)
            add(targetIndex, row)
        }

        component.value.clear()
        component.value.addAll(newRows)

        rowOrder.clear()
        rowOrder.addAll(newRows.map { it.id })
        return RowResult(row, targetIndex)
    }

    override fun up(id: String, by: Int): RowResult<Row> = moveRow(id, -by)

    override fun down(id: String, by: Int): RowResult<Row> = moveRow(id, by)
}
