package joyfill2.editors.table.internal

import cinematic.mutableLiveOf
import joyfill2.Document
import joyfill2.IdentityGenerator
import joyfill2.editors.components.ComponentEditor
import joyfill2.editors.components.internal.AbstractComponentEditorFinder
import joyfill2.editors.table.RowManager
import joyfill2.events.ChangeEvent
import joyfill2.table.Cell
import joyfill2.table.Column
import joyfill2.table.Row
import joyfill2.tools.validation.ComponentInvalid
import joyfill2.tools.validation.ComponentValid
import joyfill2.tools.validation.ComponentValidity
import joyfill2.tools.validation.RowInvalid
import joyfill2.tools.validation.RowValid
import joyfill2.tools.validation.RowValidity

internal abstract class AbstractRowManager(
    override val row: Row,
    private val document: Document,
    val columns: List<Column>,
    private val identity: IdentityGenerator,
    val onChange: ((ChangeEvent) -> Unit)?,
    val onValidate: (RowValidity) -> Unit,
    val fieldId: String,
    val identifier: String,
) : AbstractComponentEditorFinder(), RowManager {

    override fun copyInto(other: RowManager) {
        for (column in columns) copyInto(other, column)
    }

    override val isSelected = mutableLiveOf(false)

    override fun select(): RowManager = apply { isSelected.value = true }
    override fun unselect(): RowManager = apply { isSelected.value = false }

    protected val cache = mutableMapOf<String, ComponentEditor>()
    private val validation = mutableMapOf<Column, ComponentValidity>()

    private fun Cell.toEditor(): ComponentEditor = toEditor(
        cache = cache,
        validation = validation,
        document = document,
        identity = identity,
        onChange = onChange,
        fieldId = fieldId,
        identifier = identifier,
        onCellValidated = ::onCellValidated
    )

    override fun all(): List<ComponentEditor> = row.cells.all().map { it.toEditor() }
    override fun find(key: String?): ComponentEditor? = row.cells.find(key)?.toEditor()

    private fun onCellValidated(cell: Cell, validity: ComponentValidity) {
        validation[cell.column] = validity
        onValidate(validate())
    }

    private fun validate(): RowValidity {
        val valid = validation.filterValues { it is ComponentValid }
            .mapValues { it.value as ComponentValid }
        val invalid = validation.filterValues { it is ComponentInvalid }
            .mapValues { it.value as ComponentInvalid }

        val validity = if (invalid.isEmpty()) {
            RowValid(row, valid)
        } else {
            RowInvalid(row, valid + invalid, valid, invalid)
        }
        return validity
    }
}