package joyfill2.editors.multi_select.internal

import cinematic.mutableLiveOf
import joyfill2.Document
import joyfill2.collections.PageCollection
import joyfill2.editors.components.internal.AbstractComponentEditor
import joyfill2.editors.multi_select.MultiSelectEditor
import joyfill2.events.ChangeEvent
import joyfill2.events.EventDispatcher
import joyfill2.multi_select.MultiSelectComponent
import joyfill2.tools.validation.ComponentValidity
import joyfill2.tools.visibility.Visibility
import joyfill2.utils.Option

internal class MultiSelectEditorImpl(
    override val component: MultiSelectComponent,
    override val options: List<Option>,
    document: Document,
    pages: PageCollection?,
    onChange: ((ChangeEvent) -> Unit)?,
    onValidate: (ComponentValidity) -> Unit,
    identifier: String,
    fieldId: String = component.id,
    dispatcher: EventDispatcher?,
) : MultiSelectEditor,
    AbstractComponentEditor(
        document,
        pages,
        component,
        onValidate,
        onChange,
        identifier,
        fieldId,
        dispatcher
    ) {

    private fun look(key: String?): Option? =
        options.firstOrNull { it.id == key || it.value == key }

    override val state by lazy {
        mutableLiveOf(
            MultiSelectEditorStateImpl(
                data = options.filter { it.id in component.value }.toMutableList(),
                visibility = if (component.hidden) Visibility.Hidden else Visibility.Visible,
                validity = validate(),
            )
        )
    }

    override fun changeHiddenTo(value: Boolean) {
        hidden = value
        state.value = state.value.copy(
            visibility = if (value) Visibility.Hidden else Visibility.Visible,
        )
    }

    override fun canSelectMultiple(): Boolean = component.multi

    override fun select(key: String?) {
        val option = look(key) ?: return
        val selected = this.component.value.toMutableSet()
        if (selected.contains(option.id)) return
        if (!component.multi) {
            component.value.clear()
            state.value = state.value.copy(
                data = mutableListOf(),
                validity = validate()
            )
        }
        component.value.add(option.id)
        updateStateAndNotify(selected().toMutableList())
    }

    override fun customCondition(): Boolean = component.value.isEmpty()

    private fun updateStateAndNotify(data: MutableList<Option>) {
        resolveConditions()
        state.value = state.value.copy(
            data = data,
            validity = validate(),
            visibility = if (component.hidden) Visibility.Hidden else Visibility.Visible,
        )
        notifyChange(component.value)
    }

    override fun select(item: Option?): Option? {
        select(item?.id)
        return item
    }

    override fun unselect(key: String?) {
        val option = look(key) ?: return
        val selected = this.component.value.toMutableSet()
        if (!selected.contains(option.id)) return
        component.value.remove(option.id)
        updateStateAndNotify(selected().toMutableList())
    }

    override fun unselect(item: Option?): Option? {
        unselect(item?.id)
        return item
    }

    private fun selected(): List<Option> {
        val found = mutableListOf<Option>()
        val value = this.component.value
        for (option in value) found.add(look(option) ?: continue)
        return found
    }

    override fun set(options: List<Option>) {
        val pool = options.map { it.id }
        val candidates = options.filter { pool.contains(it.id) }.toSet()
        component.value.clear()
        component.value.addAll(candidates.map { it.id }.toMutableList())
        updateStateAndNotify(selected().toMutableList())
    }
}