package joyfill2.editors.file.internal

import cinematic.mutableLiveOf
import joyfill2.Document
import joyfill2.IdentityGenerator
import joyfill2.collections.PageCollection
import joyfill2.components.internal.AbstractFileComponent
import joyfill2.editors.components.internal.AbstractComponentEditor
import joyfill2.editors.file.AbstractFileEditor
import joyfill2.events.ChangeEvent
import joyfill2.events.EventDispatcher
import joyfill2.tools.validation.ComponentValidity
import joyfill2.tools.visibility.Visibility
import joyfill2.utils.Attachment
import joyfill2.utils.ID
import joyfill2.utils.toAttachment

internal abstract class AbstractFileEditorImpl<T : AbstractFileComponent>(
    override val component: T,
    private val identity: IdentityGenerator,
    document: Document,
    pages: PageCollection?,
    onChange: ((ChangeEvent) -> Unit)?,
    onValidate: (ComponentValidity) -> Unit,
    identifier: String,
    fieldId: String,
    dispatcher: EventDispatcher?,
) : AbstractComponentEditor(
    document,
    pages,
    component,
    onValidate,
    onChange,
    identifier,
    fieldId,
    dispatcher
),
    AbstractFileEditor {

    override val state by lazy {
        mutableLiveOf(
            FileEditorStateImpl(
                data = component.value,
                visibility = if (component.hidden) Visibility.Hidden else Visibility.Visible,
                validity = validate()
            )
        )
    }

    override fun changeHiddenTo(value: Boolean) {
        hidden = value
        state.value = state.value.copy(
            visibility = if (value) Visibility.Hidden else Visibility.Visible,
        )
    }

    private fun updateStateAndNotify(data: List<Attachment>) {
        resolveConditions()
        state.value = state.value.copy(
            data = data.map { it.copy() }.toMutableList(),
            validity = validate(),
            visibility = if (component.hidden) Visibility.Hidden else Visibility.Visible,
        )
        notifyChange(data.map { it.toMap() }.toMutableList())
    }

    override fun add(url: String) = add(listOf(url))
    override fun add(attachment: Attachment) = add(listOf(attachment))


    override fun add(attachments: List<Attachment>) {
        component.value.addAll(attachments)
        updateStateAndNotify(component.value)
    }

    override fun add(urls: Collection<String>) {
        val attachments = urls.map { url ->
            mutableMapOf<String, Any?>(
                ID to identity.generate(),
                Attachment::url.name to url
            ).toAttachment()
        }.toMutableList()

        component.value.addAll(attachments)
        updateStateAndNotify(component.value)
    }

    override fun set(attachments: List<Attachment>) {
        component.value.clear()
        component.value.addAll(attachments)
        updateStateAndNotify(component.value)
    }

    override fun set(urls: Collection<String>) {
        val attachments = urls.map { url ->
            mutableMapOf<String, Any?>(
                ID to identity.generate(),
                Attachment::url.name to url
            ).toAttachment()
        }.toMutableList()

        component.value.addAll(attachments)
        updateStateAndNotify(component.value)
    }

    override fun remove(attachment: Attachment?) {
        attachment?.let {
            remove(listOf(it))
        }
    }

    override fun remove(key: String?) {
        key?.let {
            remove(listOf(it))
        }
    }

    override fun customCondition(): Boolean = component.value.isEmpty()

    override fun remove(attachments: List<Attachment>) {
        val ids = attachments.map { it.id }
        val urlsList = attachments.map { it.url }
        val fileNames = attachments.mapNotNull { it.fileName }
        val found = component.value.filter {
            it.id in ids || it.fileName in fileNames || it.url in urlsList
        }
        if (!found.isNotEmpty()) return
        component.value.removeAll(found)
        updateStateAndNotify(component.value)
    }

    override fun remove(keys: Collection<String>) {
        val found = component.value.filter {
            it.id in keys || it.fileName in keys || it.url in keys
        }
        if (found.isEmpty()) return
        component.value.removeAll(found)
        updateStateAndNotify(component.value)
    }

    override fun canSelectMultiple(): Boolean = component.multi
}