package joyfill2.editors.event

import joyfill2.Document
import joyfill2.components.Component
import joyfill2.events.ChangeEvent
import joyfill2.events.ChangeLog
import joyfill2.events.Target
import joyfill2.events.internal.FieldUpdateChangeImpl
import joyfill2.events.toChangeEvent
import joyfill2.table.Column
import joyfill2.table.TableSchema
import joyfill2.utils.ID

internal open class EventTrigger<out C : Component>(
    private val document: Document,
    open val component: C,
    filedId: String,
    private val fieldIdentifier: String,
    private val onChange: ((ChangeEvent) -> Unit)?
) {
    val file by lazy { document.files[0] }

    val page by lazy {
        document.files.flatMap { it.pages }.find { page ->
            val positions = page.positions.map { it.field }
            positions.contains(filedId)
        } ?: throw IllegalStateException("Component not found in any page")
    }

    val position by lazy {
        page.positions.find { it.field == filedId }
            ?: throw IllegalStateException("Component not found in any position")
    }

    private val capturedEvents = mutableListOf<EventCapture>()

    protected fun captureEvent(
        value: Any?,
        target: String = Target.field_update
    ) {
        capturedEvents.add(EventCapture(value, target = target))
    }

    protected fun emitCapturedEvents() {
        capturedEvents.forEach { capture ->
            onChange?.invoke(createChangeEvent(capture))
        }
        capturedEvents.clear()
    }

    protected fun emitSingleEvent(
        value: Any?,
        target: String = Target.field_update
    ) {
        onChange?.invoke(createChangeEvent(EventCapture(value = value, target = target)))
    }

    protected fun notifyChange(
        value: Any?,
        target: String = Target.field_update
    ) {
        onChange?.invoke((createChangeEvent(EventCapture(value = value, target = target))))
    }

    private fun createChangeEvent(capture: EventCapture): ChangeEvent = mutableMapOf<String, Any?>(
        ChangeEvent::changelogs.name to mutableListOf(createChangeLog(capture)),
        ChangeEvent::document.name to document.toMap(),
    ).toChangeEvent(
        columns = ((capture.value as? MutableMap<String, Any?>)?.get(TableSchema::columns.name) as? List<Column>)
            ?: emptyList()
    )

    private fun createChangeLog(capture: EventCapture): MutableMap<String, Any> = mutableMapOf(
        ID to document.id,
        ChangeLog::fieldPositionId.name to position.id,
        ChangeLog::pageId.name to page.id,
        ChangeLog::fieldIdentifier.name to fieldIdentifier,
        ChangeLog::identifier.name to document.identifier,
        ChangeLog::fieldId.name to component.id,
        ChangeLog::fileId.name to file.id,
        ChangeLog::target.name to capture.target,
        ChangeLog::createdOn.name to capture.timestamp,
        ChangeLog::change.name to when (capture.target) {
            Target.field_update -> mutableMapOf(FieldUpdateChangeImpl::value.name to capture.value)
            Target.field_value_rowCreate,
            Target.field_value_rowUpdate,
            Target.field_value_rowDelete,
            Target.field_value_rowMove -> (capture.value as MutableMap<String, Any?>).toMutableMap()
                .apply { remove(TableSchema::columns.name) }

            else -> mutableMapOf(FieldUpdateChangeImpl::value.name to capture.value)
        },
        ChangeLog::sdk.name to "kotlin",
        ChangeLog::v.name to 1,
    )
}