package joyfill2.editors.dropdown.internal

import cinematic.mutableLiveOf
import joyfill2.Document
import joyfill2.collections.PageCollection
import joyfill2.dropdown.DropdownComponent
import joyfill2.editors.components.internal.AbstractComponentEditor
import joyfill2.editors.dropdown.DropdownEditor
import joyfill2.events.ChangeEvent
import joyfill2.events.EventDispatcher
import joyfill2.tools.validation.ComponentValidity
import joyfill2.tools.visibility.Visibility
import joyfill2.utils.Option

internal class DropdownEditorImpl(
    override val component: DropdownComponent,
    override val options: List<Option>,
    document: Document,
    pages: PageCollection?,
    onChange: ((ChangeEvent) -> Unit)?,
    onValidate: (ComponentValidity) -> Unit,
    identifier: String,
    fieldId: String = component.id,
    dispatcher: EventDispatcher?,
) : DropdownEditor,
    AbstractComponentEditor(document, pages,component, onValidate, onChange, identifier, fieldId, dispatcher) {

    override val state by lazy {
        mutableLiveOf(
            DropdownEditorStateImpl(
                data = options.firstOrNull { it.id == component.value },
                visibility = if (component.hidden) Visibility.Hidden else Visibility.Visible,
                validity = validate()
            )
        )
    }

    override fun changeHiddenTo(value: Boolean) {
        hidden = value
        state.value = state.value.copy(
            visibility = if (value) Visibility.Hidden else Visibility.Visible,
        )
    }

    private fun look(key: String?): Option? =
        options.firstOrNull { it.id == key || it.value == key }

    override fun select(key: String?) {
        val option = look(key) ?: return
        if (component.value == option.id) return
        component.value = option.id
        updateStateAndNotify(option)
    }

    override fun select(item: Option?): Option? {
        select(item?.id)
        return item
    }

    override fun unselect(item: Option?): Option? {
        unselect(item?.id)
        return item
    }

    override fun unselect(key: String?) {
        val option = look(key) ?: return
        if (component.value != option.id) return
        component.value = null
        updateStateAndNotify(null)
    }

    private fun updateStateAndNotify(data: Option?) {
        resolveConditions()
        state.value = state.value.copy(
            data = data,
            validity = validate(),
            visibility = if (component.hidden) Visibility.Hidden else Visibility.Visible,
        )
        notifyChange(data?.value)
    }

    override fun customCondition(): Boolean = component.value.isNullOrEmpty()

    override fun unselectAll() {
        component.value = null
        updateStateAndNotify(null)
    }
}