package joyfill2.editors.document

import joyfill2.Document
import joyfill2.IdentityGenerator
import joyfill2.Stage
import joyfill2.View
import joyfill2.collections.internal.FieldCollectionImpl
import joyfill2.collections.internal.PageCollectionImpl
import joyfill2.editors.components.AbstractCompStringEditor
import joyfill2.editors.event.EventCapture
import joyfill2.editors.number.NumberEditor
import joyfill2.editors.utils.indexMap
import joyfill2.events.ChangeEvent
import joyfill2.tools.validation.FieldsInvalid
import joyfill2.tools.validation.FieldsValid
import joyfill2.tools.validation.FieldsValidity
import joyfill2.tools.validation.Valid
import joyfill2.utils.ID
import wisdom.ResolutionResource
import wisdom.ResolutionResourceBuilder
import wisdom.parse
import wisdom.resolutionResource

internal class DocumentEditorImpl(
    private val document: Document,
    private val identity: IdentityGenerator,
    private val layout: LayoutConfig,
    private val functions: (ResolutionResourceBuilder.() -> Unit)?,
    private val onChange: ((ChangeEvent) -> Unit)?,
) : DocumentEditor {

    companion object {
        const val STAGE = "stage"
    }

    override fun toMap() = document.toMap()

    override var stage: Stage
        get() = get<String?>(STAGE)?.let { Stage.valueOf(it) } ?: Stage.draft
        set(value) = set(STAGE, value.name)

    override var name: String
        get() = document.name
        set(value) {
            document.name = value
        }

    override var id: String
        get() = document.id
        set(value) = set(ID, value)

    override var identifier: String
        get() = get(Document::identifier.name)
        set(value) = set(Document::identifier.name, value)

    private val resolver: ResolutionResource = resolutionResource {
        include(wisdom.std.logic)
        include(wisdom.std.math)
        include(wisdom.std.string)

        getter { id ->
            val field = fields.find(id)
            when (field) {
                is NumberEditor -> field.component.value
                is AbstractCompStringEditor -> field.component.value
                else -> null
            }
        }
        functions?.invoke(this)
    }

    private val library by lazy {
        val codes = document.fields.filter { it.formulas.firstOrNull() != null }.associate { field ->
            val formular = field.formulas.first()
            field.id to document.formulas.first { it.desc == formular.formula }.formula
        }
        parse(codes)
    }

    override val fields by lazy { FieldCollectionImpl(document, pages, identity, resolver, library, onChange) }

    override val pages by lazy { PageCollectionImpl(document) }

    override val capturedEvents: MutableList<EventCapture> = mutableListOf()

    override val views: List<View> get() = document.files.flatMap { it.views }

    override fun set(key: String, value: Any?) = document.set(key, value)

    override fun <R> get(key: String): R = document.get(key)


    override fun integrity(): FieldsValidity {
        TODO("Not yet implemented")
    }

    override fun resolveConditions() {
        TODO("Not yet implemented")
    }

    override fun toDocument(): Document = document

    override fun toJsonObject() = document.toJsonObject()

    override fun validate(): FieldsValidity {
        val them = pages.pages.flatMap { fields.from(it) }.associateBy { it.id }
        val sorter = them.map { it.value.id }.indexMap()

        val all = them.values.map { it.validate() }.sortedBy { sorter[it.component.id] }
        val valid = all.filter { it is Valid }.sortedBy { sorter[it.component.id] }

        return if (valid.size == all.size) {
            FieldsValid(valid)
        } else {
            FieldsInvalid(all)
        }
    }

    override fun toJsonString() = document.toJsonString()
}
