package joyfill2.editors.components.internal

import joyfill2.Field
import joyfill2.conditions.Action
import joyfill2.conditions.AndEval
import joyfill2.conditions.Contains
import joyfill2.conditions.Empty
import joyfill2.conditions.Equals
import joyfill2.conditions.Filled
import joyfill2.conditions.GreaterThan
import joyfill2.conditions.HideAction
import joyfill2.conditions.LessThan
import joyfill2.conditions.NotEquals
import joyfill2.conditions.OrEval
import joyfill2.conditions.ShowAction
import joyfill2.conditions.UnknownConditionOperator
import joyfill2.conditions.UnknownEval
import joyfill2.conditions.field.FieldConditionDefinition
import joyfill2.conditions.field.FieldLogic
import joyfill2.date.DateField
import joyfill2.dropdown.DropdownCell
import joyfill2.dropdown.DropdownField
import joyfill2.editors.components.ValueEvaluator
import joyfill2.multi_select.MultiSelectCell
import joyfill2.multi_select.MultiSelectField
import joyfill2.number.NumberField
import joyfill2.value.ListValue
import joyfill2.value.SingleValue
import kotlinx.datetime.LocalDate

internal class FieldConditionEvaluator(private val getField: (String) -> Field?) : ValueEvaluator {

    fun evaluateConditions(
        logic: FieldLogic,
        currentHidden: Boolean,
        initialHidden: Boolean,
    ): Boolean {
        val conditions = logic.conditions
        if (conditions.isEmpty()) return currentHidden

        return when (logic.eval) {
            AndEval -> isActionMet(
                logic.action,
                initialHidden
            ) { conditions.all { isConditionMet(it) } }

            OrEval -> isActionMet(
                logic.action,
                initialHidden
            ) { conditions.any { isConditionMet(it) } }

            is UnknownEval -> initialHidden
        }
    }


    private fun isActionMet(
        action: Action,
        initialHidden: Boolean,
        conditionMet: () -> Boolean,
    ): Boolean {
        return if (conditionMet() && action is ShowAction) {
            false
        } else if (conditionMet() && action is HideAction) {
            true
        } else {
            initialHidden
        }
    }

    private fun isConditionMet(condition: FieldConditionDefinition): Boolean {
        val field = getField(condition.field) ?: return false
        val v = field.value

        return when (condition.condition) {
            Filled -> field.isFilled()
            Contains -> field.hasValueContaining(condition.value)
            Empty -> isValueEmpty(v)
            Equals -> field.hasValueEqualTo(condition.value)
            NotEquals -> !field.hasValueEqualTo(condition.value)
            GreaterThan -> isGreaterThan(v, condition.value)
            LessThan -> isLessThan(v, condition.value)
            is UnknownConditionOperator -> false
        }
    }

    private fun Field.hasValueContaining(value: Any?): Boolean {
        val rhs = value
        return when (this) {
            is MultiSelectCell -> {
                val lhs = column.options.filter { it.id in this.value }
                return when (rhs) {
                    is String -> lhs.find { it.id == rhs || it.value == rhs } != null
                    is Array<*> -> rhs.all { v -> v in lhs.map { it.id } || v in lhs.map { it.value } }
                    null -> lhs.isEmpty()
                    else -> false
                }
            }

            is DropdownCell -> hasValueEqualTo(value)

            else -> this.value?.toString()?.contains(rhs.toString(), ignoreCase = true) == true
        }
    }

    private fun Field.hasValueEqualTo(rhs: Any?): Boolean {
        if (value == null && rhs == null) return true
        if (value != null && rhs == null) return false
        return when (this) {
            is MultiSelectField -> {
                val lhs = options.filter { it.id in value }
                return when (rhs) {
                    is String -> lhs.size == 1 && (lhs.first().id == rhs || lhs.first().value == rhs)
                    is Array<*> -> lhs.size == rhs.size && lhs.all { it.id in rhs || it.value in rhs }
                    null -> lhs.isEmpty()
                    else -> false
                }
            }

            is DropdownField -> {
                val lhs = options.firstOrNull { it.id == value }
                return when (rhs) {
                    is String -> lhs?.id == rhs || lhs?.value == rhs
                    is Array<*> -> rhs.size == 1 && (lhs?.id == rhs.first() || lhs?.value == rhs.first())
                    null -> lhs == null
                    else -> false
                }
            }

            is NumberField -> {
                value == rhs?.toString()?.toDoubleOrNull()
            }

            is DateField -> when (rhs) {
                is String -> value == LocalDate.parse(rhs).toEpochDays().toLong()
                is Number -> value == rhs
                else -> false
            }

            else -> when (val v = value) {
                is String -> v.contentEquals("$rhs", ignoreCase = true)
                else -> v == rhs
            }
        }
    }

    private fun Field.isFilled(): Boolean = when (this) {
        is ListValue<*> -> value.isNotEmpty()
        is SingleValue<*> -> when (val v = value) {
            is String -> v.isNotBlank()
            else -> v != null
        }

        else -> false
    }
}