package joyfill2.editors.components.internal

import joyfill2.Document
import joyfill2.Field
import joyfill2.collections.PageCollection
import joyfill2.components.Component
import joyfill2.conditions.field.FieldConditionDefinition
import joyfill2.editors.components.AbstractCompStringEditor
import joyfill2.editors.components.ComponentEditor
import joyfill2.editors.event.EventTrigger
import joyfill2.editors.number.NumberEditor
import joyfill2.events.ChangeEvent
import joyfill2.events.EventDispatcher
import joyfill2.schemas.Schema
import joyfill2.table.Cell
import joyfill2.tools.validation.ComponentInvalid
import joyfill2.tools.validation.ComponentValid
import joyfill2.tools.validation.ComponentValidity
import joyfill2.tooltip.ToolTip
import wisdom.ResolutionResource
import wisdom.ast.Library

internal abstract class AbstractComponentEditor(
    private val document: Document,
    private val pages: PageCollection?,
    override val component: Component,
    val onValidate: (ComponentValidity) -> Unit = {},
    onChange: ((ChangeEvent) -> Unit)?,
    identifier: String,
    parentFieldId: String = component.id,
    private val dispatcher: EventDispatcher? = null,
    private val dependents: List<ComponentEditor> = listOf(),
    private val resolver: ResolutionResource? = null,
    private val library: Library? = null,
) : EventTrigger<Component>(document, component, parentFieldId, identifier, onChange),
    ComponentEditor,
    Schema by component {
    override fun show() = changeHiddenTo(false)

    override fun hide() = changeHiddenTo(true)

    private val conditionEvaluator by lazy { FieldConditionEvaluator(::getField) }

    private fun getField(key: String): Field? =
        document.fields.find {
            it.identifier == key || it.id == key || it.title == key
        }

    private fun FieldConditionDefinition.isInvolved(with: Field): Boolean {
        return with.identifier == field || with.id == field || with.title == field
    }

    override fun resolveConditions() {
        if (dispatcher == null || component is Cell) return

        // resolving conditions for fields
        document.fields.filter { it.logic != null }.forEach { field ->
            val logic = field.logic ?: return@forEach
            if (logic.conditions.none { it.isInvolved(with = component as Field) }) return@forEach

            dispatcher.setHidden(
                component = field.id,
                value =
                    conditionEvaluator.evaluateConditions(
                        logic = logic,
                        currentHidden = field.hidden,
                        initialHidden = field.initialHiddenState,
                    ),
            )
        }

        // resolving conditions for pages
        pages?.raw()?.filter { it.logic != null }?.forEach { p ->
            val logic = p.logic ?: return@forEach
            if (logic.conditions.none { it.isInvolved(with = component as Field) }) return@forEach

            pages.setHidden(
                key = p.id,
                value =
                    conditionEvaluator.evaluateConditions(
                        logic = logic,
                        currentHidden = p.hidden,
                        initialHidden = p.initialHiddenState,
                    )
            )
        }
    }

    override val tip: ToolTip?
        get() =
            when (val c = component) {
                is Field -> c.tip
                else -> null
            }


    override var identifier: String
        get() =
            when (val c = component) {
                is Field -> c.identifier
                else -> c.id
            }
        set(value) {
            when (val c = component) {
                is Field -> c.identifier = value
                else -> c.id = value
            }
        }

    override val disabled: Boolean
        get() =
            when (val c = component) {
                is Field -> c.disabled
                else -> false
            }

    open fun customCondition(): Boolean = true

    override fun updateDependentsValues() {
        if (library == null) return
        for (dependent in dependents) {
            if (!library.routines.containsKey(dependent.id)) continue
            when (dependent) {
                is NumberEditor -> {
                    val result = library.call(dependent.id, resolver) as? Double
                    if (dependent.state.value.data == result) continue
                    dependent.value(result)
                }

                is AbstractCompStringEditor -> {
                    val result = library.call(dependent.id, resolver) as? String
                    if (dependent.state.value.data == result) continue
                    dependent.value(result)
                }
            }
        }
    }

    override fun validate(): ComponentValidity {
        val validity =
            when {
                component.required && customCondition() ->
                    ComponentInvalid(component, listOf("Component ${component.title} is required"))

                else ->
                    ComponentValid(component, emptyList())
            }
        onValidate(validity)
        return validity
    }
}