package joyfill2.editors.collection.internal

import joyfill2.Document
import joyfill2.IdentityGenerator
import joyfill2.collection.toNestedTableRow
import joyfill2.collections.PageCollection
import joyfill2.editors.collection.CollectionEditor
import joyfill2.editors.collection.RowManager
import joyfill2.editors.collection.TableEditor
import joyfill2.editors.table.RowResult
import joyfill2.editors.table.internal.AbstractRows
import joyfill2.editors.table.intiCells
import joyfill2.events.ChangeEvent
import joyfill2.table.Row
import joyfill2.tools.validation.RowValidity
import joyfill2.utils.ID

internal class NestedRowsImpl(
    private val host: CollectionEditor,
    private val document: Document,
    private val pages: PageCollection,
    private val table: TableEditor,
    private val identity: IdentityGenerator,
    private val next: () -> RowManager?,
    private val initialHiddenMap: Map<String, Boolean>,
    private val prev: () -> RowManager?,
    private val onAppend: (tableId: String) -> Unit,
    private val onCreate: (row: RowManager, index: Int) -> Unit,
    private val onDelete: (rows: List<RowManager>) -> Unit,
    private val onMove: (RowResult<Row>) -> Unit,
    private val onRowChange: ((ChangeEvent, Row) -> Unit)?,
    private val onChange: ((ChangeEvent) -> Unit)?,
    private val onValidate: (RowValidity) -> Unit,
    private val resolveConditions: (id: String) -> Unit,
) : AbstractRows<RowManager>(table.component, table.component.rowOrder) {

    override fun create(empty: Boolean): RowManager {
        val row = mutableMapOf<String, Any?>(
            ID to identity.generate(),
            Row::deleted.name to false,
            Row::cells.name to table.component.intiCells(empty).toMap(),
            Row::children.name to mutableMapOf<String, Any?>()
        ).toNestedTableRow(host.component.schema, table.component)
        return row.toManager()
    }

    override fun append(): RowManager {
        val result = super.append()
        val index = table.component.value.lastIndex
        onCreate(result, index)
        return result
    }

    override fun down(id: String, by: Int): RowResult<Row> {
        val result = super.down(id, by)
        onMove(result)
        return result
    }

    override fun up(id: String, by: Int): RowResult<Row> {
        val result = super.up(id, by)
        onMove(result)
        return result
    }

    override fun addAfter(id: String): RowManager? {
        val result = super.addAfter(id)
        val index = table.component.value.indexOfFirst { it.id == id } + 1
        result?.let {
            onCreate(result, index)
        }
        return result
    }

    override fun delete(keys: List<String>): List<RowManager> {
        val result = super.delete(keys)
        onDelete(result)
        return result
    }

    override fun Row.toManager(): RowManager {
        val uid = "${table.id}:$id"
        return cache.getOrPut(uid) {
            RowManagerImpl(
                id = uid,
                host = host,
                initialHiddenMap = initialHiddenMap,
                document = document,
                row = this,
                columns = table.component.columns,
                identity = identity,
                table = table.component,
                onValidate = onValidate,
                onRowChange = {
                    onRowChange?.invoke(it, this)
                },
                onChange = onChange,
                onAppend = onAppend,
                pages = pages,
                resolveConditions = resolveConditions,
                next = next,
                prev = prev,
            )
        }
    }
}