package joyfill2.editors.collection.internal

import joyfill2.Document
import joyfill2.IdentityGenerator
import joyfill2.collections.PageCollection
import joyfill2.editors.collection.CollectionEditor
import joyfill2.editors.collection.RowManager
import joyfill2.editors.collection.TableEditor
import joyfill2.editors.collection.entries.RowEntry
import joyfill2.editors.table.Columns
import joyfill2.editors.table.RowResult
import joyfill2.editors.table.internal.AbstractTableEditor
import joyfill2.editors.table.internal.ColumnsImpl
import joyfill2.events.ChangeEvent
import joyfill2.events.RowChange
import joyfill2.events.Target
import joyfill2.table.Cell
import joyfill2.table.Row
import joyfill2.table.TableComponent

internal class NestableTableEditor(
    override var id: String,
    private val host: CollectionEditor,
    override val component: TableComponent,
    override val parent: RowManager? = null,
    private val identity: IdentityGenerator,
    private val initialHiddenMap: Map<String, Boolean>,
    private val next: () -> RowManager?,
    private val prev: () -> RowManager?,
    private val onAppend: (tableId: String) -> Unit,
    private val resolveConditions: (id: String) -> Unit,
    document: Document,
    pages: PageCollection,
    onChange: ((ChangeEvent) -> Unit)?,
) : AbstractTableEditor(component, document, pages, onChange, host.identifier, initialHiddenMap, host.id),
    TableEditor {

    override fun getCell(column: String): Cell? = parent?.row?.cells?.find(column)

    override val columns: Columns by lazy { ColumnsImpl(component) }

    override val rows by lazy {
        NestedRowsImpl(
            host = host,
            document = document,
            pages = pages,
            table = this,
            identity = identity,
            next = next,
            initialHiddenMap = initialHiddenMap,
            prev = prev,
            onAppend = onAppend,
            onCreate = ::onCreate,
            onDelete = ::onDelete,
            onMove = ::onMove,
            resolveConditions = resolveConditions,
            onRowChange = ::onRowChange,
            onChange = onChange,
            onValidate = ::onRowValidated,
        )
    }

    init {
        rows.all().forEach {
            it.tables.all().forEach {
                it.resolveConditions()
            }
        }
    }

    private fun onCreate(row: RowManager, index: Int) {
        val parentPath = createPathFroChangeLogs()
        val value = buildRowCreateChangeValue(row, index).apply {
            put(RowChange::parentPath.name, parentPath)
            put(RowChange::schemaId.name, component.id)
        }
        val updatedRows = state.value.rows.toMutableList()
        updatedRows.add(index, row)

        state.value = state.value.copy(
            rows = updatedRows
        )

        row.tables.all().forEach {
            it.resolveConditions()
        }

        onAppend(id)

        notifyChange(
            value = value,
            target = Target.field_value_rowCreate
        )
    }

    private fun onRowChange(changeEvent: ChangeEvent, row: Row) {
        changeEvent.changelogs.forEach { changeLog ->
            val value = buildRowUpdateChangeValue(row, changeLog).apply {
                put(RowChange::parentPath.name, createPathFroChangeLogs())
                put(RowChange::schemaId.name, component.id)
            }
            notifyChange(
                value = value,
                target = Target.field_value_rowUpdate
            )
        }

        val rowManager = rows.find(row.id) ?: return
        val tables = rowManager.tables.all()

        if (tables.isEmpty() || tables.all { it.component.tableLogic == null }) return
        tables.forEach { it.resolveConditions() }
        resolveConditions(rowManager.id)
    }

    private fun onDelete(rows: List<RowManager>) {
        rows.forEach {
            val value = buildRowDeleteChangeValue(it).apply {
                put(RowChange::parentPath.name, createPathFroChangeLogs())
                put(RowChange::schemaId.name, component.id)
            }
            notifyChange(
                value = value,
                target = Target.field_value_rowDelete
            )
        }
    }

    private fun onMove(rowResult: RowResult<Row>) {
        val value = buildRowMoveChangeValue(rowResult.row, rowResult.index).apply {
            put(RowChange::parentPath.name, createPathFroChangeLogs())
            put(RowChange::schemaId.name, component.id)
        }
        notifyChange(
            value = value,
            target = Target.field_value_rowMove
        )
    }

    private fun createPathFroChangeLogs(): String {
        return if (parent != null) {
            val rowEntries = host.state.value.entries.filterIsInstance<RowEntry>()
            val parentRowEntry = rowEntries.firstOrNull { it.manager.row.id == parent.row.id }

            // Check if parentRowEntry is null
            if (parentRowEntry == null) return ""

            val parentIndex = (parentRowEntry.number.minus(1))

            val parentTableSchemaId = parentRowEntry.table.component.id

            if (parentRowEntry.table.parent != null) {
                val parentTableEditor = parentRowEntry.table
                if (parentTableEditor is NestableTableEditor) {
                    val ancestorPath = parentTableEditor.createPathFroChangeLogs()
                    if (ancestorPath.isNotEmpty()) {
                        "$ancestorPath.$parentIndex.$parentTableSchemaId"
                    } else {
                        "$parentIndex.$parentTableSchemaId"
                    }
                } else {
                    "$parentIndex.$parentTableSchemaId"
                }
            } else {
                "$parentIndex.$parentTableSchemaId"
            }
        } else {
            ""
        }
    }
}
