package joyfill2.editors.collection.internal

import cinematic.mutableLiveOf
import joyfill2.Document
import joyfill2.IdentityGenerator
import joyfill2.collection.CollectionComponent
import joyfill2.collections.PageCollection
import joyfill2.editors.collection.CollectionEditor
import joyfill2.editors.collection.RowManager
import joyfill2.editors.collection.TableEditor
import joyfill2.editors.collection.entries.LazyEntry
import joyfill2.editors.collection.entries.RowEntry
import joyfill2.editors.collection.entries.TableEntry
import joyfill2.editors.components.internal.AbstractComponentEditor
import joyfill2.editors.table.Selection
import joyfill2.events.ChangeEvent
import joyfill2.table.internal.TableComponentImpl
import joyfill2.tools.validation.ComponentInvalid
import joyfill2.tools.validation.ComponentValid
import joyfill2.tools.validation.ComponentValidity
import joyfill2.tools.visibility.Visibility

internal class CollectionEditorImpl(
    override val component: CollectionComponent,
    private val identity: IdentityGenerator,
    document: Document,
    pages: PageCollection,
    onChange: ((ChangeEvent) -> Unit)?,
    identifier: String,
) : AbstractComponentEditor(
    document = document,
    component = component,
    onValidate = {},
    onChange = onChange,
    identifier = identifier,
    pages = pages,
), CollectionEditor {

    override fun changeHiddenTo(value: Boolean) {
        hidden = value
    }

    private val initialHiddenMap =
        component.schema.all().associate {
            it.id to it.hidden
        }.toMap()

    private val cached by lazy {
        val schema = component.schema.root() ?: error("Root table not found")
        schema.required = component.required
        NestableTableEditor(
            id = schema.id,
            host = this,
            component = TableComponentImpl(component, schema),
            parent = null,
            identity = identity,
            next = { next() },
            prev = { prev() },
            onAppend = ::onAppend,
            resolveConditions = { id ->
                val entry = state.value.entries.find { it.id == id } as? RowEntry
                entry?.let {
                    if (it.expanded) {
                        expand(it)
                    }
                }
            },
            document = document,
            pages = pages,
            onChange = onChange,
            initialHiddenMap = initialHiddenMap,
        )
    }


    private fun onAppend(tableId: String) {
        val tableEntries = state.value.entries
            .filterIsInstance<TableEntry>()

        val tableEntry = tableEntries
            .firstOrNull { it.table.id == tableId } ?: return

        expand(tableEntry)
        state.value = state.value.copy(
            validity = validate()
        )
    }

    override fun root(): TableEditor = cached

    override val state by lazy {
        val root = root()
        mutableLiveOf(
            CollectionEditorStateImpl(
                entries = TableEntry(
                    level = 0,
                    expanded = true,
                    table = root,
                    hidden = false,
                ).expanded(),
                selected = emptyList(),
                validity = validate(),
                form = null,
                visibility = if (hidden) Visibility.Hidden else Visibility.Visible,
            )
        )
    }

    override fun validate(): ComponentValidity = when {
        component.required && component.value.isEmpty() ->
            ComponentInvalid(component = component, messages = listOf("Invalid add at least one row to the root table"))

        else ->
            ComponentValid(component = component, messages = listOf())
    }

    override fun addAfter() {
        val selected = state.value.selected
        if (selected.size == 1) {
            val selectedRow = selected.first()
            val selectedTableId = selectedRow.table.id
            val table = state.value.entries.find { it.id == selectedTableId } as TableEntry
            selectedRow.table.rows.addAfter(selectedRow.manager.row.id)
            expand(table)
            state.value = state.value.copy(
                validity = validate()
            )
            unselectAll()
        }
    }

    override fun moveUp() {
        val selected = state.value.selected
        if (selected.size == 1) {
            val selectedRow = selected.first()
            val selectedTableId = selectedRow.table.id
            val table = state.value.entries.find { it.id == selectedTableId } as TableEntry
            table.table.rows.up(selectedRow.manager.row.id)
            expand(table)
        }
    }

    override fun moveDown() {
        val selected = state.value.selected
        if (selected.size == 1) {
            val selectedRow = selected.first()
            val selectedTableId = selectedRow.table.id
            val table = state.value.entries.find { it.id == selectedTableId } as TableEntry
            table.table.rows.down(selectedRow.manager.row.id)
            expand(table)
        }
    }

    override fun addFormAfter() {
        val form = state.value.form ?: return close()
        val formTableId = form.table.id
        val table = state.value.entries.find { it.id == formTableId } as TableEntry
        val createdRowManger = form.table.rows.addAfter(form.manager.row.id)

        expand(table)
        val addedRowEntry = state.value
            .entries
            .filterIsInstance<RowEntry>()
            .firstOrNull {
                it.manager.row.id == createdRowManger?.row?.id
            } ?: return
        state.value = state.value.copy(
            form = addedRowEntry
        )
    }

    override fun submit() {
        val form = state.value.form ?: return close()
        state.value.selected.forEach {
            form.manager.copyInto(it.manager)
        }
        close()
    }


    override fun toggle(entry: LazyEntry) = when (entry) {
        is RowEntry -> if (entry.expanded) collapse(entry) else expand(entry)
        is TableEntry -> if (entry.expanded) collapse(entry) else expand(entry)
    }

    override fun select(item: RowEntry?): RowEntry? {
        item ?: return null
        if (item.table.id != state.value.selected.firstOrNull()?.table?.id) {
            unselectAll()
        }
        val selected = state.value.selected
        if (item in selected) return null
        state.value = state.value.copy(selected = selected + item)
        return item
    }

    override fun unselect(item: RowEntry?): RowEntry? {
        item ?: return null
        val selected = state.value.selected
        if (item !in selected) return null
        state.value = state.value.copy(selected = selected - item)

        return item
    }

    override fun unselectAll() {
        state.value = state.value.copy(selected = emptyList())
    }

    override fun edit() {
        val selected = state.value.selected
        if (selected.isEmpty()) return close()
        val first = selected.first()
        if (selected.size > 1) {
            val newRowManager = first.table.rows.create(true)
            val newRowEntry = first.copy(manager = newRowManager)
            state.value = state.value.copy(form = newRowEntry)
        } else {
            state.value = state.value.copy(form = first)
        }
    }

    override fun selectAll(entry: TableEntry) {
        unselectAll()
        val selectedRowEntryList = state.value.entries.filterIsInstance<RowEntry>()
            .filter { it.table.id == entry.table.id }
        state.value = state.value.copy(
            selected = selectedRowEntryList
        )
    }

    override fun delete() {
        val candidates = state.value.selected
        for (candidate in candidates) {
            candidate.table.rows.delete(listOf(candidate.manager.row.id))
        }
        state.value = state.value.copy(
            entries = state.value.entries.filterNot { it in candidates },
            selected = emptyList(),
            validity = validate(),
        )
        val parentTable = candidates.first().table
        val tableEntry =
            state.value.entries.firstOrNull { it.table.id == parentTable.id } as? TableEntry
        tableEntry?.let {
            expand(it)
        }
    }

    override fun close() {
        state.value = state.value.copy(form = null)
    }

    private fun prev(): RowManager? {
        val currentForm = state.value.form ?: return null
        val tableRows = state.value.entries.filterIsInstance<RowEntry>()
            .filter { it.table.id == currentForm.table.id }
        val prevNumber = currentForm.number - 1
        val entry = tableRows.firstOrNull { it.number == prevNumber } ?: return null
        state.value = state.value.copy(form = entry)
        return entry.manager
    }

    private fun next(): RowManager? {
        val currentForm = state.value.form ?: return null
        val tableRows = state.value.entries.filterIsInstance<RowEntry>()
            .filter { it.table.id == currentForm.table.id }
        val nextNumber = currentForm.number + 1
        val entry = tableRows.firstOrNull { it.number == nextNumber } ?: return null
        state.value = state.value.copy(form = entry)
        return entry.manager
    }

    override fun selection(entry: TableEntry): Selection {
        val selected = state.value.selected.filter { entry.table.id == it.table.id }
        val rows = entry.table.rows.all()
        return when {
            selected.isEmpty() -> Selection.None
            selected.size == rows.size -> Selection.All
            selected.isEmpty() -> Selection.None
            selected.isNotEmpty() -> Selection.Some(selected.map { it.manager })
            else -> Selection.None
        }
    }
}
