package joyfill2.collections.internal

import cinematic.mutableLiveOf
import joyfill2.Document
import joyfill2.IdentityGenerator
import joyfill2.Page
import joyfill2.collections.PageCollection
import joyfill2.collections.PageCollectionState
import joyfill2.conditions.AndEval
import joyfill2.conditions.HideAction
import joyfill2.conditions.OrEval
import joyfill2.conditions.ShowAction
import joyfill2.editors.page.PageEditor
import joyfill2.editors.page.PageEditorImpl
import joyfill2.toField
import joyfill2.toPage

internal class PageCollectionImpl(
    private val document: Document,
    private val page: String? = null
): PageCollection {
    private val files by lazy { document.files }

    private val views by lazy { files.flatMap { it.views } }

    private val view by lazy { views.find { it.type == "mobile" } }

    val pages by lazy { view?.pages ?: files.flatMap { it.pages } }

    override val state by lazy { mutableLiveOf(computePageState(page)) }

    private fun computePageState(key: String?): PageCollectionState {
        val them = pages.mapNotNull { find(it.id) }.filter { !it.page.hidden }.map { it.page }
        val current = find(key)?.page ?: them.first()
        return PageCollectionState(view, them, current)
    }

    override fun raw(): List<Page> = files.flatMap { it.pages }

    override fun all() = files.flatMap { it.pages }.map {
        cache.getOrPut(it.id) { PageEditorImpl(document, it) }
    }

    private val cache = mutableMapOf<String, PageEditor>()
    override fun find(key: String?): PageEditor? {
        val k = key ?: return null
        val page = pages.find {
            it.id == k || it.name == k || it.identifier == k
        } ?: return null
        return cache.getOrPut(page.id) { PageEditorImpl(document, page) }
    }

    override fun at(index: Int): PageEditor? {
        if (index < 0) return null
        val pages = files.flatMap { it.pages }
        if (index >= pages.size) return null
        val page = pages[index]
        return cache.getOrPut(page.id) { PageEditorImpl(document, page) }
    }

    override fun setHidden(key: String?, value: Boolean): Page? {
        val editor = find(key) ?: return null
        val p = editor.page
        editor.page.hidden = value
        state.value = computePageState(state.value.page.id)
        return p
    }

    override fun navigate(page: Page?): Page? {
        val p = page ?: return null
        state.value = state.value.copy(page = p)
        return p
    }

    override fun navigate(page: String?): Page? = navigate(find(page)?.page)

    override fun duplicate(page: String?, newName: String): Page? {
        val p = find(page)?.page
        return duplicate(p, newName)
    }

    override fun duplicate(page:Page?, newName: String): Page? {
        if (page == null) return null

        val mapped = page.toMap().toMutableMap()
        val id = IdentityGenerator.default.generate()
        mapped["_id"] = id
        mapped["name"] = newName
        //regenerate field positions ids
        mapped["fieldPositions"] =  (mapped["fieldPositions"] as List<Map<String, Any>>? ?: emptyList() ).map {
            it.toMutableMap().apply { this["_id"] = IdentityGenerator.default.generate() }
        }

        val fieldInPositions  = page.positions.map { it.field }.toSet()
        val oldToNewFieldsIds = mutableMapOf<String, String>().apply {
            fieldInPositions.forEach { put(it, IdentityGenerator.default.generate()) }
        }

        val fields = document.fields
            .filter { it.id in fieldInPositions }
            .map { f ->
                val mappedField = f.toMap().toMutableMap()
                //update this field id in the positions
                mapped["fieldPositions"] = (mapped["fieldPositions"] as List<Map<String, Any>>? ?: emptyList() ).map {
                    if (it["field"] == f.id) {
                        it.toMutableMap().apply { this["field"] = oldToNewFieldsIds[f.id] as Any }
                    } else {
                        it
                    }
                }

                //add to doc fields
                mappedField["_id"] = oldToNewFieldsIds[f.id]
                if(f.logic != null){
                    val conditions = f.logic!!.conditions.map {
                        if (it.field in oldToNewFieldsIds.keys) {
                            val condMap = it.toMap().toMutableMap()
                            condMap["field"] = oldToNewFieldsIds[it.field]
                            condMap
                        } else {
                            it.toMap()
                        }
                    }
                    mappedField["logic"] = mutableMapOf<String, Any?>().apply {
                        put("hidden", f.logic?.hidden)
                        put("action", when(f.logic?.action){
                            ShowAction -> "show"
                            HideAction -> "hide"
                            else -> "unknown"
                        })
                        put( "eval", when(f.logic?.eval){
                            AndEval -> "and"
                            OrEval -> "or"
                            else -> "unknown"
                        })

                        put("conditions", conditions)
                    }
                }
                mappedField.toField()
            }
        document.fields.addAll(fields)
        document.files.first().apply {
            val p = mapped.toPage()
            pages.add(p)
            pageOrder.add(p.id)
        }

        val newPage = mapped.toPage()
        state.value = state.value.copy(pages = state.value.pages + newPage, page = newPage)
        return newPage
    }
}