package joyfill2.collections.internal

import joyfill2.Document
import joyfill2.Field
import joyfill2.IdentityGenerator
import joyfill2.Page
import joyfill2.collections.FieldCollection
import joyfill2.collections.PageCollection
import joyfill2.editors.collection.CollectionEditor
import joyfill2.editors.components.ComponentEditor
import joyfill2.editors.components.internal.AbstractComponentEditorFinder
import joyfill2.editors.components.internal.toEditor
import joyfill2.editors.table.TableEditor
import joyfill2.events.ChangeEvent
import joyfill2.events.EventDispatcher
import wisdom.ResolutionResource
import wisdom.ast.Library

internal class FieldCollectionImpl(
    private val document: Document,
    private val pages: PageCollection,
    private val identity: IdentityGenerator,
    private val resolver: ResolutionResource,
    private val library: Library,
    private val onChange: ((ChangeEvent) -> Unit)?,
) : AbstractComponentEditorFinder(), FieldCollection, EventDispatcher {

    override fun all() = document.fields.map { it.toEditor() }

    override fun from(page: String): List<ComponentEditor> {
        val files = document.files
        val pages = files.flatMap { it.views }.flatMap { it.pages } + files.flatMap { it.pages }
        val p = pages.find { it.identifier == page || it.id == page || it.name == page }
            ?: return emptyList()
        return from(p)
    }

    private fun Field.keys() = listOf(identifier, id, title).map { it.lowercase() }

    private fun look(key: String): Field? = document.fields.find { key.lowercase() in it.keys() }

    override fun from(page: Page): List<ComponentEditor> {
        val positions = page.positions
        val ids = positions.map { it.field }
        return document.fields.filter {
            it.id in ids
        }.sortedWith(
            compareBy(
                { df -> positions.first { it.field == df.id }.y },
                { df -> positions.first { it.field == df.id }.x }
            )
        ).map { it.toEditor() }
    }

    override fun find(key: String?): ComponentEditor? {
        val field = look(key ?: return null)
        return field?.toEditor()
    }


    private fun Field.dependents(): List<ComponentEditor> {
        val fields = document.fields
        return buildList {
            for (field in fields) {
                val formulas = field.formulas.mapNotNull { application ->
                    document.formulas.find { it.desc == application.formula }
                }

                if (formulas.any { it.formula.contains(id) }) {
                    add(field.toEditor())
                }
            }
        }
    }

    private val cache = mutableMapOf<String, ComponentEditor>()

    init {
        all().forEach { it.resolveConditions() }
    }

    private fun Field.toEditor(): ComponentEditor = toEditor(
        cache = cache,
        document = document,
        identity = identity,
        resolver = resolver,
        pages = pages,
        library = library,
        onChange = onChange,
        dispatcher = this@FieldCollectionImpl,
        dependents = { dependents() }
    )

    override fun table(key: String?): TableEditor? = find(key) as? TableEditor

    override fun collection(key: String?): CollectionEditor? = find(key) as? CollectionEditor

    override fun hide(component: String) {
        find(component)?.hide()
    }

    override fun show(component: String) {
        find(component)?.show()
    }

    override fun setHidden(component: String, value: Boolean) {
        find(component)?.changeHiddenTo(value)
    }
}
