package joyfill.table.internal

import joyfill.ChangeEvent
import joyfill.Document
import joyfill.IdentityGenerator
import joyfill.fields.Field
import joyfill.fields.TableField
import joyfill.fields.table.Column
import joyfill.fields.table.DropdownColumn
import joyfill.fields.table.ImageColumn
import joyfill.fields.table.Row
import joyfill.fields.table.TextColumn
import joyfill.table.CellEditor
import joyfill.table.DropdownCellEditor
import joyfill.table.ImageCellEditor
import joyfill.table.RowEditor
import joyfill.table.TextCellEditor

internal class RowEditorImpl(
    val document: Document,
    val field: TableField,
    val identity: IdentityGenerator,
    override val row: Row,
    val onChange: ((ChangeEvent) -> Unit)?
) : RowEditor {

    private fun <C : Column> find(key: String?, type: Field.Type) = field.columns.find {
        (it.id == key || it.title == key) && it.type == type
    } as? C

    private fun <C : Column> take(index: Int, type: Field.Type): C? {
        val column = field.columns.getOrNull(index) ?: return null
        if (column.type != type) return null
        return column as? C
    }

    private fun id(column: Column) = "${row.id}:${column.id}"
    override fun text(key: String): TextCellEditor? {
        val column = find<TextColumn>(key, Field.Type.text) ?: return null
        return cache.getOrPut(id(column)) { TextCellEditorImpl(document, field, column, row, onChange) } as TextCellEditor
    }

    override fun text(index: Int): TextCellEditor? {
        val column = take<TextColumn>(index, Field.Type.text) ?: return null
        return cache.getOrPut(id(column)) { TextCellEditorImpl(document, field, column, row, onChange) } as TextCellEditor
    }

    override fun image(key: String): ImageCellEditor? {
        val column = find<ImageColumn>(key, Field.Type.image) ?: return null
        return cache.getOrPut(id(column)) { ImageCellEditorImpl(document, field, identity, column, row, onChange) } as ImageCellEditor
    }

    override fun image(index: Int): ImageCellEditor? {
        val column = take<ImageColumn>(index, Field.Type.image) ?: return null
        return cache.getOrPut(id(column)) { ImageCellEditorImpl(document, field, identity, column, row, onChange) } as ImageCellEditor
    }

    override fun dropdown(key: String): DropdownCellEditor? {
        val column = find<DropdownColumn>(key, Field.Type.dropdown) ?: return null
        return cache.getOrPut(id(column)) { DropdownCellEditorImpl(document, field, column, row, onChange) } as DropdownCellEditor
    }

    override fun dropdown(index: Int): DropdownCellEditor? {
        val column = take<DropdownColumn>(index, Field.Type.dropdown) ?: return null
        return cache.getOrPut(id(column)) { DropdownCellEditorImpl(document, field, column, row, onChange) } as DropdownCellEditor
    }

    private fun Column.toEditor(): CellEditor? = when (this) {
        is TextColumn -> cache.getOrPut(id(this)) { TextCellEditorImpl(document, field, this, row, onChange) }
        is ImageColumn -> cache.getOrPut(id(this)) { ImageCellEditorImpl(document, field, identity, this, row, onChange) }
        is DropdownColumn -> cache.getOrPut(id(this)) { DropdownCellEditorImpl(document, field, this, row, onChange) }
        else -> null
    }

    override fun col(key: String) = field.columns.find {
        it.id == key || it.title == key
    }?.toEditor()

    override fun col(index: Int) = field.columns.getOrNull(index)?.toEditor()

    override fun copyInto(other: RowEditor) {
        for (column in field.columns) when (column) {
            is TextColumn -> {
                val value = text(column.id)?.text?.value
                if (value.isNullOrBlank()) return
                other.text(column.id)?.set(value)
            }

            is DropdownColumn -> {
                val selected = dropdown(column.id)?.selected()
                val cell = other.dropdown(column.id) ?: return
                if (selected != null) {
                    cell.select(selected)
                } else {
                    cell.unselect()
                }
            }

            is ImageColumn -> {
                val images = image(column.id)?.value ?: return
                other.image(column.id)?.add(images.map { it.url })
            }

            else -> {}
        }
    }

    private val cache = mutableMapOf<String, CellEditor>()
}