package joyfill.table.internal

import cinematic.mutableLiveOf
import joyfill.editors.internal.TableFieldEditorImpl
import joyfill.fields.table.Column
import joyfill.fields.table.DropdownColumn
import joyfill.fields.table.TextColumn
import joyfill.table.ColumnCollection
import joyfill.table.ColumnFilter
import joyfill.table.ColumnState
import joyfill.table.SortingOption
import joyfill.utils.Option2

internal class ColumnCollectionImpl(
    private val editor: TableFieldEditorImpl
) : ColumnCollection {

    override val state by lazy { mutableLiveOf(ColumnState(null, emptyList())) }

    private val table by lazy { editor.field }

    override fun all(): List<Column> = table.columns

    override fun find(key: String?): Column? {
        val needle = key ?: return null
        return table.columns.firstOrNull { it.id == needle || it.title == needle }
    }

    override fun select(key: String?): Column? {
        val column = find(key) ?: return null
        return select(column)
    }

    private fun TextColumn.createFilter() = ColumnFilter.TextColumnFilter(this, SortingOption.None, null)
    private fun DropdownColumn.createFilter() = ColumnFilter.DropdownColumnFilter(this, SortingOption.None, null)

    private fun Column.createFilter(): ColumnFilter? = when (this) {
        is TextColumn -> createFilter()
        is DropdownColumn -> createFilter()
        else -> null
    }

    override fun select(column: Column?): Column? {
        val current = state.value.current?.column
        val filter = state.value.filters.find { it.column.id == column?.id }
        if (column?.id != current?.id) {
            val next = when (column) {
                is TextColumn -> filter ?: column.createFilter()
                is DropdownColumn -> filter ?: column.createFilter()
                else -> return null
            }
            state.value = if (filter == null) {
                ColumnState(current = next, state.value.filters + next)
            } else {
                state.value.copy(current = next)
            }
        } else {
            state.value = state.value.copy(current = null)
        }
        applyFilterChoices()
        return column
    }

    override fun sort(option: SortingOption?) {
        val current = state.value.current ?: return
        val next = current.sorted(option)
        state.value = ColumnState(
            current = next,
            filters = state.value.filters - current + next
        )
        applyFilterChoices()
    }

    override fun sort() {
        val current = state.value.current ?: return
        val order = listOf(SortingOption.None, SortingOption.Ascending, SortingOption.Descending)
        val index = (order.indexOf(current.sort) + 1) % order.size
        sort(order[index])
    }

    override fun filter(option: Option2?) {
        val current = state.value.current ?: return
        val filter = current as? ColumnFilter.DropdownColumnFilter ?: return
        val next = if (option == null && current.sort == SortingOption.None) {
            current.column.createFilter()
        } else {
            filter.copy(value = option)
        }
        state.value = ColumnState(
            current = next,
            filters = state.value.filters - current + next
        )
        applyFilterChoices()
    }

    override fun filter(key: String?) {
        val current = state.value.current ?: return
        val filter = current as? ColumnFilter.TextColumnFilter ?: return
        val next = if (key.isNullOrBlank() && current.sort == SortingOption.None) {
            current.column.createFilter()
        } else {
            filter.copy(value = key)
        }
        state.value = ColumnState(
            current = next,
            filters = state.value.filters - current + next
        )
        applyFilterChoices()
    }

    private fun applyFilterChoices() = with(editor.rows) {
        state.value.haystack.applyFilters()
    }

    override fun areFiltersApplied() = all().any { state.value.isFiltered(it) }

    override fun clearFilter() {
        val current = state.value.current ?: return
        val next = current.column.createFilter() ?: return
        state.value = state.value.copy(
            current = next,
            filters = state.value.filters - current + next
        )
        applyFilterChoices()
    }
}
