package joyfill.internal

import cinematic.mutableLiveOf
import joyfill.Document
import joyfill.MutablePage
import joyfill.Page
import joyfill.collections.PageCollection
import joyfill.collections.PageCollectionState
import joyfill.editors.PageEditor

internal class PageCollectionImpl(
    private val document: Document,
    private val page: String? = null
) : PageCollection {

    private val files by lazy { document.files }

    private val views by lazy { files.flatMap { it.views } }

    private val view by lazy { views.find { it.type == "mobile" } }

    val pages by lazy { view?.pages ?: files.flatMap { it.pages } }

    override val state by lazy { mutableLiveOf(computePageState(page)) }

    private fun computePageState(key: String?): PageCollectionState {
        val them = pages.mapNotNull { find(it.id) }.filter { !it.hidden.value }.map { it.page }
        val current = find(key)?.page ?: them.first()
        return PageCollectionState(view, them, current)
    }

    override fun raw(): List<Page> = files.flatMap { it.pages }

    override fun all() = files.flatMap { it.pages }.map {
        cache.getOrPut(it.id) { PageEditorImpl(document, it as MutablePage) }
    }

    private val cache = mutableMapOf<String, PageEditor>()
    override fun find(key: String?): PageEditor? {
        val k = key ?: return null
        val page = pages.find {
            it.id == k || it.name == k || it.identifier == k
        } ?: return null
        return cache.getOrPut(page.id) { PageEditorImpl(document, page as MutablePage) }
    }

    override fun at(index: Int): PageEditor? {
        if (index < 0) return null
        val pages = files.flatMap { it.pages }
        if (index >= pages.size) return null
        val page = pages[index]
        return cache.getOrPut(page.id) { PageEditorImpl(document, page as MutablePage) }
    }

    override fun show(key: String?): Page? = setHidden(key, false)

    override fun hide(key: String?): Page? = setHidden(key, true)

    override fun setHidden(key: String?, value: Boolean): Page? {
        val editor = find(key) ?: return null
        val p = editor.page
        editor.hidden.value = value
        state.value = computePageState(state.value.page.id)
        return p
    }

    override fun navigate(page: Page?): Page? {
        val p = page ?: return null
        state.value = state.value.copy(page = p)
        return p
    }

    override fun navigate(page: String?): Page? = navigate(find(page)?.page)
}