package joyfill.internal

import joyfill.ChangeEvent
import joyfill.EventDispatcher
import joyfill.IdentityGenerator
import joyfill.MutableDocument
import joyfill.Page
import joyfill.collections.FieldCollection
import joyfill.collections.PageCollection
import joyfill.editors.BarcodeFieldEditor
import joyfill.editors.ChartFieldEditor
import joyfill.editors.DateFieldEditor
import joyfill.editors.DropdownFieldEditor
import joyfill.editors.FieldEditor
import joyfill.editors.FileFieldEditor
import joyfill.editors.ImageFieldEditor
import joyfill.editors.MultiSelectFieldEditor
import joyfill.editors.NumberFieldEditor
import joyfill.editors.SignatureFieldEditor
import joyfill.editors.TableFieldEditor
import joyfill.editors.TextAreaFieldEditor
import joyfill.editors.TextFieldEditor
import joyfill.editors.internal.AbstractValueBasedFieldEditor
import joyfill.editors.internal.BarcodeFieldEditorImpl
import joyfill.editors.internal.BlockFieldEditorImpl
import joyfill.editors.internal.ChartFieldEditorImpl
import joyfill.editors.internal.DateFieldEditorImpl
import joyfill.editors.internal.DropdownFieldEditorImpl
import joyfill.editors.internal.FileFieldEditorImpl
import joyfill.editors.internal.ImageFieldEditorImpl
import joyfill.editors.internal.MultiSelectFieldEditorImpl
import joyfill.editors.internal.NumberFieldEditorImpl
import joyfill.editors.internal.RichTextFieldEditorImpl
import joyfill.editors.internal.SignatureFieldEditorImpl
import joyfill.editors.internal.TableFieldEditorImpl
import joyfill.editors.internal.TextAreaFieldEditorImpl
import joyfill.editors.internal.TextFieldEditorImpl
import joyfill.editors.internal.UnknownFieldEditor
import joyfill.fields.BarcodeField
import joyfill.fields.BlockField
import joyfill.fields.ChartField
import joyfill.fields.DateField
import joyfill.fields.DropdownField
import joyfill.fields.Field
import joyfill.fields.Field.Type
import joyfill.fields.Field.Type.chart
import joyfill.fields.Field.Type.date
import joyfill.fields.Field.Type.dropdown
import joyfill.fields.Field.Type.file
import joyfill.fields.Field.Type.image
import joyfill.fields.Field.Type.multiSelect
import joyfill.fields.Field.Type.number
import joyfill.fields.Field.Type.signature
import joyfill.fields.Field.Type.table
import joyfill.fields.Field.Type.text
import joyfill.fields.Field.Type.textarea
import joyfill.fields.Field.Type.barcode
import joyfill.fields.FileField
import joyfill.fields.ImageField
import joyfill.fields.MultiSelectField
import joyfill.fields.NumberField
import joyfill.fields.RichTextField
import joyfill.fields.SignatureField
import joyfill.fields.TableField
import joyfill.fields.TextAreaField
import joyfill.fields.TextField
import joyfill.fields.VField

internal class FieldCollectionImpl(
    private val document: MutableDocument,
    private val identity: IdentityGenerator,
    private val pages: PageCollection,
    private val onChange: ((ChangeEvent) -> Unit)?
) : FieldCollection, EventDispatcher {

    override fun all() = document.fields.map { it.toEditor() }

    override fun from(page: String): List<FieldEditor> {
        val files = document.files
        val pages = files.flatMap { it.views }.flatMap { it.pages } + files.flatMap { it.pages }
        val p = pages.find { it.identifier == page || it.id == page || it.name == page } ?: return emptyList()
        return from(p)
    }

    override fun from(page: Page): List<FieldEditor> {
        val positions = page.positions
        val ids = positions.map { it.field }
        return document.fields.filter {
            it.id in ids
        }.sortedWith(
            compareBy(
                { df -> positions.first { it.field == df.id }.y },
                { df -> positions.first { it.field == df.id }.x }
            )
        ).map { it.toEditor() }
    }

    private fun <F : Field> look(key: String, type: Type): F? = document.fields.find {
        (it.identifier == key || it.id == key || it.title == key) && it.type == type
    } as? F

    override fun find(key: String): FieldEditor? {
        val field = document.fields.find { it.identifier == key || it.id == key || it.title == key } ?: return null
        return field.toEditor()
    }

    private val cache = mutableMapOf<String, FieldEditor>()

    init {
        all().forEach { it.resolveConditions() }
    }

    private fun VField.toEditor(): FieldEditor = cache.getOrPut(id) {
        when (this) {
            is TextField -> TextFieldEditorImpl(document, this, pages, onChange, this@FieldCollectionImpl)
            is TextAreaField -> TextAreaFieldEditorImpl(document, this, pages, onChange, this@FieldCollectionImpl)
            is NumberField -> NumberFieldEditorImpl(document, this, pages, onChange, this@FieldCollectionImpl)
            is DropdownField -> DropdownFieldEditorImpl(document, this, pages, this@FieldCollectionImpl, onChange)
            is MultiSelectField -> MultiSelectFieldEditorImpl(document, this, pages, this@FieldCollectionImpl, onChange)
            is DateField -> DateFieldEditorImpl(document, this, pages, onChange, this@FieldCollectionImpl)
            is SignatureField -> SignatureFieldEditorImpl(document, this, pages, onChange, this@FieldCollectionImpl)
            is TableField -> TableFieldEditorImpl(document, this, identity, pages, this@FieldCollectionImpl, onChange)
            is ChartField -> ChartFieldEditorImpl(document, this, identity, pages, this@FieldCollectionImpl, onChange)
            is ImageField -> ImageFieldEditorImpl(document, this, identity, pages, this@FieldCollectionImpl, onChange)
            is FileField -> FileFieldEditorImpl(document, this, identity, pages, this@FieldCollectionImpl, onChange)
            is RichTextField -> RichTextFieldEditorImpl(document, this, pages, this@FieldCollectionImpl)
            is BlockField -> BlockFieldEditorImpl(document, this, pages, this@FieldCollectionImpl)
            is BarcodeField -> BarcodeFieldEditorImpl(document, this, pages, onChange, this@FieldCollectionImpl)
            else /*  is UnknownField */ -> UnknownFieldEditor(document, this, pages, this@FieldCollectionImpl, onChange)
        }
    }

    override fun text(key: String) = look<TextField>(key, text)?.toEditor() as? TextFieldEditor

    override fun textarea(key: String) = look<TextAreaField>(key, textarea)?.toEditor() as? TextAreaFieldEditor

    override fun signature(key: String) = look<SignatureField>(key, signature)?.toEditor() as? SignatureFieldEditor

    override fun date(key: String) = look<DateField>(key, date)?.toEditor() as? DateFieldEditor

    override fun number(key: String) = look<NumberField>(key, number)?.toEditor() as? NumberFieldEditor

    override fun dropdown(key: String) = look<DropdownField>(key, dropdown)?.toEditor() as? DropdownFieldEditor

    override fun select(key: String) = look<MultiSelectField>(key, multiSelect)?.toEditor() as? MultiSelectFieldEditor

    override fun image(key: String) = look<ImageField>(key, image)?.toEditor() as? ImageFieldEditor

    override fun file(key: String) = look<FileField>(key, file)?.toEditor() as? FileFieldEditor

    override fun barcode(key: String) = look<BarcodeField>(key, barcode)?.toEditor() as? BarcodeFieldEditor

    override fun table(key: String) = look<TableField>(key, table)?.toEditor() as? TableFieldEditor

    override fun chart(key: String) = look<ChartField>(key, chart)?.toEditor() as? ChartFieldEditor

    override fun hide(field: String) {
        find(field)?.hide()
    }

    override fun show(field: String) {
        find(field)?.show()
    }

    override fun setHidden(field: String, value: Boolean) {
        find(field)?.setHidden(value)
    }
}