package joyfill.internal

import joyfill.ChangeEvent
import joyfill.Document
import joyfill.IdentityGenerator
import joyfill.JoyStage
import joyfill.MutableDocument
import joyfill.Valid
import joyfill.Validation
import joyfill.View
import joyfill.editors.DocumentEditor
import joyfill.toMutableDocument
import joyfill.utils.ID
import joyfill.validation.DocumentInvalid
import joyfill.validation.DocumentValid
import joyfill.validation.DocumentValidity
import joyfill.validation.FieldValid

internal class DocumentEditorImpl(
    private val document: MutableDocument,
    private val identity: IdentityGenerator,
    private val onChange: ((ChangeEvent) -> Unit)?,
) : DocumentEditor {

    constructor(
        document: Document,
        identity: IdentityGenerator,
        onChange: ((ChangeEvent) -> Unit)?,
    ) : this(document.toMutableDocument(), identity, onChange)

    companion object {
        const val STAGE = "stage"
    }

    override var stage: JoyStage
        get() = get<String?>(STAGE)?.let { JoyStage.valueOf(it) } ?: JoyStage.draft
        set(value) = set(STAGE, value.name)

    override var name: String
        get() = document.name
        set(value) {
            document.name = value
        }

    override var id: String
        get() = document.id
        set(value) = set(ID, value)

    override var identifier: String
        get() = get(Document::identifier.name)
        set(value) = set(Document::identifier.name, value)

    override val pages by lazy { PageCollectionImpl(document) }

    override val fields by lazy { FieldCollectionImpl(document, identity, pages, onChange) }

    override val views: List<View> get() = document.files.flatMap { it.views }

    override fun set(key: String, value: Any?) = document.set(key, value)

    override fun <R> get(key: String): R = document.get(key)

    override fun integrity(): Validation {
        return Valid(emptyList())
    }

    override fun validate(): Validation {
        return Valid(emptyList())
    }

    override fun validateDocument(): DocumentValidity {
        val them = pages.pages.flatMap { fields.from(it) }.associateBy { it.id }
        val all = them.values.map { it.validate() }
        val valid = all.filterIsInstance<FieldValid>()
        return if (valid.size == all.size) {
            DocumentValid(valid)
        } else {
            DocumentInvalid(all)
        }
    }

    override fun resolveConditions() {

    }

    override fun toMap() = document.toMap()

    override fun toDocument(): Document = document

    override fun toJsonObject() = document.toJsonObject()

    override fun toJsonString() = document.toJsonString()
}