package joyfill.editors.internal

import cinematic.mutableLiveOf
import joyfill.ChangeEvent
import joyfill.Document
import joyfill.EventDispatcher
import joyfill.collections.PageCollection
import joyfill.conditions.AndEval
import joyfill.conditions.ConditionDefinition
import joyfill.conditions.Contains
import joyfill.conditions.Empty
import joyfill.conditions.Equals
import joyfill.conditions.Filled
import joyfill.conditions.GreaterThan
import joyfill.conditions.HideAction
import joyfill.conditions.LessThan
import joyfill.conditions.NotEquals
import joyfill.conditions.OrEval
import joyfill.conditions.ShowAction
import joyfill.conditions.UnknownConditionOperator
import joyfill.conditions.UnknownEval
import joyfill.editors.FieldEditor
import joyfill.fields.DateField
import joyfill.fields.DropdownField
import joyfill.fields.Field
import joyfill.fields.ListBasedField
import joyfill.fields.MultiSelectField
import joyfill.fields.NumberField
import joyfill.fields.VField
import joyfill.fields.ValueBasedField
import kotlinx.datetime.LocalDate

@PublishedApi
internal abstract class AnyFieldEditor<out F : VField>(
    document: Document,
    override val field: F,
    private val pages: PageCollection,
    private val dispatcher: EventDispatcher,
    onChange: ((ChangeEvent) -> Unit)?
) : EventTrigger<F>(document, field, onChange), FieldEditor {

    override var id: String
        get() = this.field.id
        set(value) {
            this.field.id = value
        }

    override var identifier: String
        get() = this.field.identifier
        set(value) {
            this.field.identifier = value
        }

    override var title: String
        get() = this.field.title
        set(value) {
            this.field.title = value
        }

    override val type get() = field.type

    override val hidden by lazy { mutableLiveOf(field.hidden) }

    override fun show() = setHidden(false)

    override fun hide() = setHidden(true)

    override fun setHidden(value: Boolean) {
        hidden.value = value
    }

    override fun resolveConditions() {
        // Conditional Logic on Fields
        document.fields.filter { it.logic != null }.forEach { f ->
            val logic = f.logic ?: return@forEach

            if (logic.conditions.none { it.isInvolved(with = field) }) return@forEach

            val conditions = logic.conditions

            if (conditions.isEmpty()) return@forEach

            val hidden = f.hidden
            if (logic.action is ShowAction && !hidden) return@forEach
            if (logic.action is HideAction && hidden) return@forEach

            val eval = logic.eval

            when (eval) {
                AndEval -> if (conditions.all { it.isMet() }) when (logic.action) {
                    is ShowAction -> dispatcher.show(f.id)
                    is HideAction -> dispatcher.hide(f.id)
                    else -> dispatcher.setHidden(f.id, hidden)
                } else dispatcher.setHidden(f.id, hidden)

                OrEval -> if (conditions.any { it.isMet() }) when (logic.action) {
                    is ShowAction -> dispatcher.show(f.id)
                    is HideAction -> dispatcher.hide(f.id)
                    else -> dispatcher.setHidden(f.id, hidden)
                } else dispatcher.setHidden(f.id, hidden)

                is UnknownEval -> {}
            }
        }

        // Conditional Logic on Pages

        pages.raw().filter { it.logic != null }.forEach { p ->

            val logic = p.logic ?: return@forEach

            if (logic.conditions.none { it.isInvolved(with = field) }) return@forEach

            val conditions = logic.conditions

            if (conditions.isEmpty()) return@forEach

            val hidden = p.hidden

            if (logic.action is ShowAction && !hidden) return@forEach
            if (logic.action is HideAction && hidden) return@forEach

            val eval = logic.eval

            when (eval) {
                AndEval -> if (conditions.all { it.isMet() }) when (logic.action) {
                    is ShowAction -> pages.show(p.id)
                    is HideAction -> pages.hide(p.id)
                    else -> pages.setHidden(p.id, hidden)
                } else pages.setHidden(p.id, hidden)

                OrEval -> if (conditions.any { it.isMet() }) when (logic.action) {
                    is ShowAction -> pages.show(p.id)
                    is HideAction -> pages.hide(p.id)
                    else -> pages.setHidden(p.id, hidden)
                } else pages.setHidden(p.id, hidden)

                is UnknownEval -> {}
            }
        }
    }

    private fun ConditionDefinition.isMet(): Boolean {
        val f = document.fields.find {
            it.identifier == field || it.id == field || it.title == field
        } ?: return false

        val v = f.value

        return when (condition) {
            Filled -> f.isFilled()
            Contains -> f.hasValueContaining(value)

            Empty -> when (v) {
                null -> true
                is String -> v.isBlank()
                is Number -> false
                is Boolean -> false
                is Array<*> -> v.isEmpty()
                is Map<*, *> -> v.isEmpty()
                is Collection<*> -> v.isEmpty()
                else -> false
            }

            Equals -> f.hasValueEqualTo(value)
            NotEquals -> !f.hasValueEqualTo(value)
            GreaterThan -> {
                val n1 = v as? Number ?: return false
                val n2 = value as? Number ?: return false
                return n1.toDouble() > n2.toDouble()
            }

            LessThan -> {
                val n1 = v as? Number ?: return false
                val n2 = value as? Number ?: return false
                return n1.toDouble() < n2.toDouble()
            }

            is UnknownConditionOperator -> false
        }
    }

    private fun VField.hasValueContaining(value: Any?): Boolean {
        val rhs = value
        return when (this) {
            is MultiSelectField -> {
                val lhs = options.filter { it.id in this.value }
                return when (rhs) {
                    is String -> lhs.find { it.id == rhs || it.value == rhs } != null
                    is Array<*> -> rhs.all { v -> v in lhs.map { it.id } || v in lhs.map { it.value } }
                    null -> lhs.isEmpty()
                    else -> false
                }
            }

            is DropdownField -> hasValueEqualTo(value)

            else -> this.value?.toString()?.contains(rhs.toString(), ignoreCase = true) == true
        }
    }

    private fun VField.isFilled(): Boolean = when (this) {
        is ListBasedField<*> -> value.isNotEmpty()
        is ValueBasedField<*> -> when (val v = value) {
            is String -> v.isNotBlank()
            else -> v != null
        }

        else -> false
    }

    private fun VField.hasValueEqualTo(rhs: Any?): Boolean {
        if (value == null && rhs == null) return true
        if (value != null && rhs == null) return false
        return when (this) {
            is MultiSelectField -> {
                val lhs = options.filter { it.id in value }
                return when (rhs) {
                    is String -> lhs.size == 1 && (lhs.first().id == rhs || lhs.first().value == rhs)
                    is Array<*> -> lhs.size == rhs.size && lhs.all { it.id in rhs || it.value in rhs }
                    null -> lhs.isEmpty()
                    else -> false
                }
            }

            is DropdownField -> {
                val lhs = options.firstOrNull { it.id == value }
                return when (rhs) {
                    is String -> lhs?.id == rhs || lhs?.value == rhs
                    is Array<*> -> rhs.size == 1 && (lhs?.id == rhs.first() || lhs?.value == rhs.first())
                    null -> lhs == null
                    else -> false
                }
            }

            is NumberField -> {
                value == rhs.toString().toDoubleOrNull()
            }

            is DateField -> when (rhs) {
                is String -> value == LocalDate.parse(rhs).toEpochDays().toLong()
                is Number -> value == rhs
                else -> false
            }

            else -> when (val v = value) {
                is String -> v.contentEquals("$rhs", ignoreCase = true)
                else -> v == rhs
            }
        }
    }

    private fun ConditionDefinition.isInvolved(with: Field): Boolean {
        return with.identifier == field || with.id == field || with.title == field
    }
}