/*
 * Decompiled with CFR 0.152.
 */
package io.indextables.tantivy4java.batch;

import io.indextables.tantivy4java.batch.BatchDocument;
import io.indextables.tantivy4java.core.IndexWriter;
import io.indextables.tantivy4java.util.Facet;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.StandardCharsets;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

public class BatchDocumentBuilder
implements AutoCloseable {
    private static final int MAGIC_NUMBER = 1413566036;
    private static final int INITIAL_CAPACITY = 8192;
    private final List<BatchDocument> documents = new ArrayList<BatchDocument>();
    private int estimatedSize = 0;
    private IndexWriter associatedWriter = null;
    private boolean closed = false;

    public BatchDocumentBuilder() {
    }

    public BatchDocumentBuilder(IndexWriter writer) {
        this.associatedWriter = writer;
    }

    public BatchDocumentBuilder addDocument(BatchDocument document) {
        if (this.closed) {
            throw new IllegalStateException("Cannot add documents to a closed BatchDocumentBuilder");
        }
        if (document == null) {
            throw new IllegalArgumentException("Document cannot be null");
        }
        this.documents.add(document);
        this.estimatedSize += this.estimateDocumentSize(document);
        return this;
    }

    public BatchDocumentBuilder addDocuments(List<BatchDocument> documents) {
        if (documents == null) {
            throw new IllegalArgumentException("Documents list cannot be null");
        }
        for (BatchDocument doc : documents) {
            this.addDocument(doc);
        }
        return this;
    }

    public BatchDocumentBuilder addDocumentFromMap(Map<String, Object> fields) {
        BatchDocument doc = BatchDocument.fromMap(fields);
        return this.addDocument(doc);
    }

    public int getDocumentCount() {
        return this.documents.size();
    }

    public int getEstimatedSize() {
        return this.estimatedSize + 16;
    }

    public int getCurrentSize() {
        if (this.documents.isEmpty()) {
            return 0;
        }
        int estimatedSize = this.getEstimatedSize() + 4 + this.documents.size() * 4 + 12;
        ByteBuffer tempBuffer = ByteBuffer.allocateDirect((int)((double)estimatedSize * 1.2));
        tempBuffer.order(ByteOrder.nativeOrder());
        try {
            this.serializeBatch(tempBuffer);
            return tempBuffer.position();
        }
        catch (Exception e) {
            return estimatedSize;
        }
    }

    public boolean wouldExceedSize(int maxSize, BatchDocument additionalDocument) {
        int additionalSize;
        if (this.documents.isEmpty()) {
            return false;
        }
        int currentSize = this.getEstimatedSize();
        return currentSize + (additionalSize = this.estimateDocumentSize(additionalDocument)) > maxSize;
    }

    public String getFormattedSize() {
        int size = this.getEstimatedSize();
        return BatchDocumentBuilder.formatByteSize(size);
    }

    public BatchMemoryStats getMemoryStats() {
        return new BatchMemoryStats(this);
    }

    public BatchDocumentBuilder clear() {
        this.documents.clear();
        this.estimatedSize = 0;
        return this;
    }

    public boolean isEmpty() {
        return this.documents.isEmpty();
    }

    public ByteBuffer build() {
        if (this.documents.isEmpty()) {
            throw new IllegalStateException("No documents to serialize");
        }
        int actualSize = this.getCurrentSize();
        if (actualSize == 0) {
            int protocolOverhead = 4 + this.documents.size() * 4 + 12;
            actualSize = Math.max(this.getEstimatedSize() + protocolOverhead, 8192);
            actualSize = (int)((double)actualSize * 1.2);
        }
        ByteBuffer buffer = ByteBuffer.allocateDirect(actualSize);
        buffer.order(ByteOrder.nativeOrder());
        try {
            this.serializeBatch(buffer);
            buffer.flip();
            return buffer;
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to serialize document batch", e);
        }
    }

    public byte[] buildArray() {
        ByteBuffer buffer = this.build();
        byte[] array = new byte[buffer.remaining()];
        buffer.get(array);
        return array;
    }

    private void serializeBatch(ByteBuffer buffer) {
        buffer.putInt(1413566036);
        ArrayList<Integer> offsets = new ArrayList<Integer>(this.documents.size());
        for (BatchDocument document : this.documents) {
            offsets.add(buffer.position());
            this.serializeDocument(buffer, document);
        }
        int offsetTableStart = buffer.position();
        Iterator iterator = offsets.iterator();
        while (iterator.hasNext()) {
            int offset = (Integer)iterator.next();
            buffer.putInt(offset);
        }
        buffer.putInt(offsetTableStart);
        buffer.putInt(this.documents.size());
        buffer.putInt(1413566036);
    }

    private void serializeDocument(ByteBuffer buffer, BatchDocument document) {
        Map<String, List<BatchDocument.FieldValue>> fields = document.getAllFields();
        if (fields.size() > 65535) {
            throw new RuntimeException("Too many fields in document: " + fields.size());
        }
        buffer.putShort((short)fields.size());
        for (Map.Entry<String, List<BatchDocument.FieldValue>> entry : fields.entrySet()) {
            String fieldName = entry.getKey();
            List<BatchDocument.FieldValue> values = entry.getValue();
            this.serializeField(buffer, fieldName, values);
        }
    }

    private void serializeField(ByteBuffer buffer, String fieldName, List<BatchDocument.FieldValue> values) {
        byte[] nameBytes = fieldName.getBytes(StandardCharsets.UTF_8);
        if (nameBytes.length > 65535) {
            throw new RuntimeException("Field name too long: " + fieldName);
        }
        buffer.putShort((short)nameBytes.length);
        buffer.put(nameBytes);
        BatchDocument.FieldType fieldType = values.get(0).getType();
        buffer.put(fieldType.getCode());
        if (values.size() > 65535) {
            throw new RuntimeException("Too many values for field: " + fieldName);
        }
        buffer.putShort((short)values.size());
        for (BatchDocument.FieldValue value : values) {
            this.serializeFieldValue(buffer, value);
        }
    }

    private void serializeFieldValue(ByteBuffer buffer, BatchDocument.FieldValue fieldValue) {
        Object value = fieldValue.getValue();
        BatchDocument.FieldType type = fieldValue.getType();
        switch (type) {
            case TEXT: 
            case JSON: 
            case IP_ADDR: {
                String strValue = value.toString();
                byte[] strBytes = strValue.getBytes(StandardCharsets.UTF_8);
                buffer.putInt(strBytes.length);
                buffer.put(strBytes);
                break;
            }
            case INTEGER: 
            case UNSIGNED: {
                buffer.putLong(((Number)value).longValue());
                break;
            }
            case FLOAT: {
                buffer.putDouble(((Number)value).doubleValue());
                break;
            }
            case BOOLEAN: {
                buffer.put((byte)((Boolean)value != false ? 1 : 0));
                break;
            }
            case DATE: {
                LocalDateTime date = (LocalDateTime)value;
                long epochMilli = date.toInstant(ZoneOffset.UTC).toEpochMilli();
                buffer.putLong(epochMilli);
                break;
            }
            case BYTES: {
                byte[] bytes = (byte[])value;
                buffer.putInt(bytes.length);
                buffer.put(bytes);
                break;
            }
            case FACET: {
                String facetPath = value instanceof Facet ? value.toString() : value.toString();
                byte[] facetBytes = facetPath.getBytes(StandardCharsets.UTF_8);
                buffer.putInt(facetBytes.length);
                buffer.put(facetBytes);
                break;
            }
            default: {
                throw new RuntimeException("Unsupported field type: " + String.valueOf((Object)type));
            }
        }
    }

    private int estimateDocumentSize(BatchDocument document) {
        int size = 2;
        for (Map.Entry<String, List<BatchDocument.FieldValue>> entry : document.getAllFields().entrySet()) {
            String fieldName = entry.getKey();
            List<BatchDocument.FieldValue> values = entry.getValue();
            size += 2 + fieldName.getBytes(StandardCharsets.UTF_8).length + 1 + 2;
            for (BatchDocument.FieldValue value : values) {
                size += this.estimateValueSize(value);
            }
        }
        return size;
    }

    private int estimateValueSize(BatchDocument.FieldValue fieldValue) {
        BatchDocument.FieldType type = fieldValue.getType();
        Object value = fieldValue.getValue();
        switch (type) {
            case TEXT: 
            case JSON: 
            case IP_ADDR: 
            case FACET: {
                return 4 + value.toString().getBytes(StandardCharsets.UTF_8).length;
            }
            case INTEGER: 
            case UNSIGNED: 
            case FLOAT: 
            case DATE: {
                return 8;
            }
            case BOOLEAN: {
                return 1;
            }
            case BYTES: {
                return 4 + ((byte[])value).length;
            }
        }
        return 32;
    }

    private static String formatByteSize(long bytes) {
        if (bytes < 1024L) {
            return String.format("%,d bytes", bytes);
        }
        if (bytes < 0x100000L) {
            return String.format("%.1f KB", (double)bytes / 1024.0);
        }
        if (bytes < 0x40000000L) {
            return String.format("%.1f MB", (double)bytes / 1048576.0);
        }
        return String.format("%.1f GB", (double)bytes / 1.073741824E9);
    }

    public BatchDocumentBuilder associateWith(IndexWriter writer) {
        this.associatedWriter = writer;
        return this;
    }

    public long[] flush() {
        if (this.associatedWriter == null) {
            return new long[0];
        }
        if (this.documents.isEmpty()) {
            return new long[0];
        }
        try {
            long[] opstamps = this.associatedWriter.addDocumentsBatch(this);
            this.clear();
            return opstamps;
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to flush batch documents to IndexWriter", e);
        }
    }

    @Override
    public void close() {
        if (this.closed) {
            return;
        }
        try {
            this.flush();
        }
        finally {
            this.closed = true;
        }
    }

    public boolean isClosed() {
        return this.closed;
    }

    public boolean hasAssociatedWriter() {
        return this.associatedWriter != null;
    }

    public static class BatchMemoryStats {
        private final int documentCount;
        private final int estimatedSize;
        private final int averageDocumentSize;
        private final int maxDocumentSize;
        private final int minDocumentSize;
        private final String formattedSize;

        BatchMemoryStats(BatchDocumentBuilder builder) {
            this.documentCount = builder.getDocumentCount();
            this.estimatedSize = builder.getEstimatedSize();
            this.formattedSize = BatchDocumentBuilder.formatByteSize(this.estimatedSize);
            if (this.documentCount > 0) {
                this.averageDocumentSize = this.estimatedSize / this.documentCount;
                int minSize = Integer.MAX_VALUE;
                int maxSize = Integer.MIN_VALUE;
                for (BatchDocument doc : builder.documents) {
                    int docSize = builder.estimateDocumentSize(doc);
                    minSize = Math.min(minSize, docSize);
                    maxSize = Math.max(maxSize, docSize);
                }
                this.minDocumentSize = minSize;
                this.maxDocumentSize = maxSize;
            } else {
                this.averageDocumentSize = 0;
                this.minDocumentSize = 0;
                this.maxDocumentSize = 0;
            }
        }

        public int getDocumentCount() {
            return this.documentCount;
        }

        public int getEstimatedSize() {
            return this.estimatedSize;
        }

        public int getAverageDocumentSize() {
            return this.averageDocumentSize;
        }

        public int getMaxDocumentSize() {
            return this.maxDocumentSize;
        }

        public int getMinDocumentSize() {
            return this.minDocumentSize;
        }

        public String getFormattedSize() {
            return this.formattedSize;
        }

        public String getFormattedAverageDocumentSize() {
            return BatchDocumentBuilder.formatByteSize(this.averageDocumentSize);
        }

        public double getDocumentsPerMB() {
            if (this.averageDocumentSize == 0) {
                return 0.0;
            }
            return 1048576.0 / (double)this.averageDocumentSize;
        }

        public String getSummary() {
            StringBuilder sb = new StringBuilder();
            sb.append("Batch Memory Statistics:\n");
            sb.append(String.format("  Documents: %,d\n", this.documentCount));
            sb.append(String.format("  Total Size: %s (%,d bytes)\n", this.formattedSize, this.estimatedSize));
            if (this.documentCount > 0) {
                sb.append(String.format("  Avg Doc Size: %s (%,d bytes)\n", this.getFormattedAverageDocumentSize(), this.averageDocumentSize));
                sb.append(String.format("  Min Doc Size: %,d bytes\n", this.minDocumentSize));
                sb.append(String.format("  Max Doc Size: %,d bytes\n", this.maxDocumentSize));
                sb.append(String.format("  Docs per MB: %.1f\n", this.getDocumentsPerMB()));
            }
            return sb.toString();
        }

        public String toString() {
            return this.getSummary();
        }
    }
}

