/*
 * Decompiled with CFR 0.152.
 */
package io.hyperfoil.core.util;

import io.hyperfoil.api.BenchmarkExecutionException;
import io.hyperfoil.internal.Properties;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BooleanSupplier;
import java.util.function.Consumer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class CpuWatchdog
implements Runnable {
    private static final Logger log = LogManager.getLogger(CpuWatchdog.class);
    private static final Path PROC_STAT = Path.of("/proc/stat", new String[0]);
    private static final long PERIOD = Properties.getLong((String)"io.hyperfoil.cpu.watchdog.period", (long)5000L);
    private static final double IDLE_THRESHOLD = Double.parseDouble(Properties.get((String)"io.hyperfoil.cpu.watchdog.idle.threshold", (String)"0.2"));
    private static final long TICK_NANOS = Properties.getLong((String)"io.hyperfoil.clock.tick.nanos", (long)10000000L);
    private final Consumer<Throwable> errorHandler;
    private final Thread thread;
    private final BooleanSupplier warmupTest;
    private final int nCpu;
    private final long[] idleTime;
    private volatile boolean running = true;
    private long lastTimestamp;
    private long now;
    private final Map<String, PhaseRecord> phaseStart = new HashMap<String, PhaseRecord>();
    private final Map<String, String> phaseUsage = new HashMap<String, String>();

    public CpuWatchdog(Consumer<Throwable> errorHandler, BooleanSupplier warmupTest) {
        this.errorHandler = errorHandler;
        this.warmupTest = warmupTest;
        File stat = PROC_STAT.toFile();
        if (!(stat.exists() && stat.isFile() && stat.canRead())) {
            log.warn("Not starting CPU watchdog as {} is not available (exists: {}, file: {}, readable: {})", (Object)PROC_STAT, (Object)stat.exists(), (Object)stat.isFile(), (Object)stat.canRead());
            this.thread = null;
            this.nCpu = 0;
            this.idleTime = null;
            return;
        }
        this.thread = new Thread((Runnable)this, "cpu-watchdog");
        this.thread.setDaemon(true);
        AtomicInteger counter = new AtomicInteger();
        this.nCpu = this.readProcStat(ignored -> counter.incrementAndGet()) ? counter.get() : 0;
        this.idleTime = new long[this.nCpu];
    }

    @Override
    public void run() {
        if (this.nCpu <= 0) {
            log.error("Illegal number of CPUs");
            return;
        }
        this.now = this.lastTimestamp = System.nanoTime();
        while (this.running) {
            if (!this.readProcStat(this::processCpuLine)) {
                log.info("CPU watchdog is terminating.");
                return;
            }
            try {
                Thread.sleep(PERIOD);
            }
            catch (InterruptedException e) {
                log.error("CPU watchdog thread interrupted, terminating.", (Throwable)e);
                return;
            }
            this.lastTimestamp = this.now;
            this.now = System.nanoTime();
        }
    }

    private boolean readProcStat(Consumer<String[]> consumer) {
        try {
            for (String line : Files.readAllLines(PROC_STAT)) {
                String[] parts;
                if (!line.startsWith("cpu") || "cpu".equals((parts = line.split(" "))[0]) || parts.length < 5) continue;
                consumer.accept(parts);
            }
            return true;
        }
        catch (IOException e) {
            log.error("CPU watchdog cannot read {}", (Object)PROC_STAT, (Object)e);
            return false;
        }
        catch (NumberFormatException e) {
            log.error("CPU watchdog cannot parse stats.", (Throwable)e);
            return false;
        }
    }

    private void processCpuLine(String[] parts) {
        double idleRatio;
        int cpuIndex = Integer.parseInt(parts[0], 3, parts[0].length(), 10);
        long idle = Long.parseLong(parts[4]);
        long prevIdle = this.idleTime[cpuIndex];
        if (prevIdle != 0L && prevIdle != Long.MAX_VALUE && this.lastTimestamp != this.now && (idleRatio = (double)(TICK_NANOS * (idle - prevIdle)) / (double)(this.now - this.lastTimestamp)) < IDLE_THRESHOLD) {
            String message = String.format("%s | CPU %d was used for %.0f%% which is more than the threshold of %.0f%%", new SimpleDateFormat("HH:mm:ss.SSS").format(new Date()), cpuIndex, 100.0 * (1.0 - idleRatio), 100.0 * (1.0 - IDLE_THRESHOLD));
            log.warn(message);
            if (this.warmupTest.getAsBoolean()) {
                this.errorHandler.accept((Throwable)new BenchmarkExecutionException(message));
                idle = Long.MAX_VALUE;
            }
        }
        if (prevIdle != Long.MAX_VALUE) {
            this.idleTime[cpuIndex] = idle;
        }
    }

    public void start() {
        if (this.thread != null) {
            this.thread.start();
        }
    }

    public void stop() {
        this.running = false;
    }

    public synchronized void notifyPhaseStart(String name) {
        if (this.nCpu <= 0) {
            return;
        }
        PhaseRecord record = new PhaseRecord(System.nanoTime(), new long[this.nCpu]);
        if (this.readProcStat(parts -> {
            int cpuIndex = Integer.parseInt(parts[0], 3, parts[0].length(), 10);
            record.cpuIdle[cpuIndex] = Long.parseLong(parts[4]);
        })) {
            this.phaseStart.putIfAbsent(name, record);
        }
    }

    public synchronized void notifyPhaseEnd(String name) {
        if (this.nCpu <= 0) {
            return;
        }
        PhaseRecord start = this.phaseStart.get(name);
        if (start == null || this.phaseUsage.containsKey(name)) {
            return;
        }
        long now = System.nanoTime();
        SumMin acc = new SumMin();
        if (this.readProcStat(parts -> {
            int cpuIndex = Integer.parseInt(parts[0], 3, parts[0].length(), 10);
            long idle = Long.parseLong(parts[4]);
            long diff = idle - start.cpuIdle[cpuIndex];
            acc.sum += diff;
            acc.min = Math.min(acc.min, diff);
        })) {
            double idleCores = (double)(TICK_NANOS * acc.sum) / (double)(now - start.timestamp);
            double minIdleRatio = (double)(TICK_NANOS * acc.min) / (double)(now - start.timestamp);
            this.phaseUsage.put(name, String.format("%.1f%% (%.1f/%d cores), 1 core max %.1f%%", 100.0 - 100.0 * idleCores / (double)this.nCpu, (double)this.nCpu - idleCores, this.nCpu, 100.0 - 100.0 * minIdleRatio));
        }
    }

    public String getCpuUsage(String name) {
        return this.phaseUsage.get(name);
    }

    private static class PhaseRecord {
        final long timestamp;
        final long[] cpuIdle;

        private PhaseRecord(long timestamp, long[] cpuIdle) {
            this.timestamp = timestamp;
            this.cpuIdle = cpuIdle;
        }
    }

    private static class SumMin {
        long sum;
        long min = Long.MAX_VALUE;

        private SumMin() {
        }
    }
}

