/*
Copyright 2021 Fausto Spoto

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package io.hotmoka.patricia.api;

import java.util.Optional;

import io.hotmoka.annotations.Immutable;

/**
 * A Merkle-Patricia trie.
 * 
 * @param <Key> the type of the keys of the trie
 * @param <Value> the type of the values of the trie
 * @param <T> the type of the trie itself
 */
@Immutable
public interface PatriciaTrie<Key, Value, T extends PatriciaTrie<Key, Value, T>> {

	/**
	 * Yields the value bound to the given key.
	 * 
	 * @param key the key
	 * @return the value, if any
	 */
	Optional<Value> get(Key key);

	/**
	 * Binds the given key to the given value. It replaces it if it was already present.
	 * This trie is not modified, but a new trie is returned instead, identical to this but for the added binding.
	 * 
	 * @param key the key
	 * @param value the value
	 * @return the resulting, modified Patricia trie
	 */
	T put(Key key, Value value);

	/**
	 * Yields the root of the trie, that can be used as a hash of its content.
	 * 
	 * @return the root
	 */
	byte[] getRoot();

	/**
	 * Yields an independent clone of this trie, but for its root, that is set to the provided value.
	 * 
	 * @param root the root to use in the cloned trie
	 * @return the resulting, cloned trie
	 * @throws UnknownKeyException if {@code root} is unknown in the store of the trie
	 */
	T checkoutAt(byte[] root) throws UnknownKeyException;
}