/*
 * Copyright (c) 2019, 2023 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.helidon.microprofile.security;

import java.lang.System.Logger.Level;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;

import io.helidon.common.context.Contexts;
import io.helidon.config.Config;
import io.helidon.microprofile.cdi.RuntimeStart;
import io.helidon.microprofile.server.JaxRsCdiExtension;
import io.helidon.microprofile.server.ServerCdiExtension;
import io.helidon.security.AuthenticationResponse;
import io.helidon.security.ProviderRequest;
import io.helidon.security.Security;
import io.helidon.security.providers.abac.AbacProvider;
import io.helidon.security.spi.AuthenticationProvider;
import io.helidon.security.spi.AuthorizationProvider;
import io.helidon.webserver.security.SecurityFeature;
import io.helidon.webserver.security.SecurityFeatureConfig;

import jakarta.annotation.Priority;
import jakarta.enterprise.context.ApplicationScoped;
import jakarta.enterprise.context.Initialized;
import jakarta.enterprise.event.Observes;
import jakarta.enterprise.inject.spi.BeanManager;
import jakarta.enterprise.inject.spi.BeforeBeanDiscovery;
import jakarta.enterprise.inject.spi.Extension;

import static jakarta.interceptor.Interceptor.Priority.LIBRARY_BEFORE;
import static jakarta.interceptor.Interceptor.Priority.PLATFORM_BEFORE;

/**
 * Extension to register bean {@link SecurityProducer}.
 */
public class SecurityCdiExtension implements Extension {
    private static final System.Logger LOGGER = System.getLogger(SecurityCdiExtension.class.getName());

    private final AtomicReference<Security> security = new AtomicReference<>();

    private Security.Builder securityBuilder = Security.builder();
    private Config config;
    private SecurityFeatureConfig.Builder securityFeatureBuilder = SecurityFeature.builder();

    /**
     * Public constructor required by service loader.
     */
    public SecurityCdiExtension() {
    }

    /**
     * Other extensions may update security builder.
     *
     * @return security builder
     */
    public Security.Builder securityBuilder() {
        if (null == securityBuilder) {
            throw new IllegalStateException("Security is already built, you cannot update the builder");
        }
        return securityBuilder;
    }

    /**
     * Access to security instance once it is created from {@link #securityBuilder()}.
     *
     * @return a security instance, or empty if not yet created
     */
    public Optional<Security> security() {
        return Optional.ofNullable(security.get());
    }

    private void registerBean(@Observes BeforeBeanDiscovery abd) {
        abd.addAnnotatedType(SecurityProducer.class, "helidon-security-producer")
                .add(ApplicationScoped.Literal.INSTANCE);
    }

    // priority is high, so we update builder from config as soon as possible
    // all additions by other extension will override configuration options
    private void configure(@Observes @RuntimeStart @Priority(PLATFORM_BEFORE + 2) Config config) {
        this.config = config;
        this.securityBuilder.config(config.get("security"));
        this.securityFeatureBuilder.config(config.get("server.features.security"));
    }

    // security must have priority higher than metrics, openapi and health
    // so we can protect these endpoints
    private void registerSecurity(@Observes @Priority(LIBRARY_BEFORE) @Initialized(ApplicationScoped.class) Object adv,
                                  BeanManager bm) {

        if (securityBuilder.noProvider(AuthenticationProvider.class)) {
            LOGGER.log(Level.INFO,
                    "Authentication provider is missing from security configuration, but security extension for microprofile "
                            + "is enabled (requires providers configuration at key security.providers). "
                            + "Security will not have any valid authentication provider");

            securityBuilder.addAuthenticationProvider(this::failingAtnProvider);
        }

        if (securityBuilder.noProvider(AuthorizationProvider.class)) {
            LOGGER.log(Level.INFO,
                    "Authorization provider is missing from security configuration, but security extension for microprofile "
                            + "is enabled (requires providers configuration at key security.providers). "
                            + "ABAC provider is configured for authorization.");
            securityBuilder.addAuthorizationProvider(AbacProvider.create());
        }

        Security tmpSecurity = securityBuilder.build();
        // free it and make sure we fail if somebody wants to update security afterwards
        securityBuilder = null;

        if (!tmpSecurity.enabled()) {
            // security is disabled, we need to set up some basic stuff - injection, security context etc.
            LOGGER.log(Level.INFO, "Security is disabled.");
            tmpSecurity = Security.builder()
                    .enabled(false)
                    .build();
        }

        // we need an effectively final instance to use in lambda
        Security security = tmpSecurity;

        // security is available in global
        Contexts.globalContext().register(security);

        JaxRsCdiExtension jaxrs = bm.getExtension(JaxRsCdiExtension.class);
        ServerCdiExtension server = bm.getExtension(ServerCdiExtension.class);
        server.addFeature(securityFeatureBuilder.build());
        securityFeatureBuilder = null;

        Contexts.context().ifPresent(ctx -> ctx.register(security));

        Config jerseyConfig = config.get("security.jersey");
        if (jerseyConfig.get("enabled").asBoolean().orElse(true)) {
            JerseySecurityFeature feature = JerseySecurityFeature.builder(security)
                    .config(jerseyConfig)
                    .build();

            if (LOGGER.isLoggable(Level.TRACE)) {
                LOGGER.log(Level.TRACE, "Security feature config: {0}. Registered for applications: {1}",
                           feature.featureConfig(),
                           jaxrs.applicationsToRun());
            }

            jaxrs.applicationsToRun()
                    .forEach(app -> app.resourceConfig().register(feature));
        } else {
            SecurityDisabledFeature feature = new SecurityDisabledFeature(security);
            jaxrs.applicationsToRun().forEach(app -> app.resourceConfig().register(feature));
        }

        this.security.set(security);
    }

    private AuthenticationResponse failingAtnProvider(ProviderRequest request) {
        return AuthenticationResponse.failed("No provider configured");
    }
}
