package geekbytes.sbt

import SBT.Module
import sbt.Append
import sbt.librarymanagement.DependencyBuilders.{
  Organization,
  OrganizationArtifactName
}
import sbt.librarymanagement.ModuleID

import scala.language.implicitConversions

case class DependencyArtifact(
    resolve: Organization => OrganizationArtifactName,
    update: Option[ModuleID => ModuleID] = None
) {
  def apply(fn: ModuleID => ModuleID): DependencyArtifact = update match {
    case Some(update) => copy(update = Some(update.andThen(fn)))
    case _            => copy(update = Some(fn))
  }
}

trait DependenciesHelper {

  case class Dependencies[Collection <: DependencyGroup](
      dependency: Collection,
      modules: Seq[Collection => DependencyArtifact]
  ) extends Utils.Builder[Seq[ModuleID]] {
    def modules(
        modules: (Collection => DependencyArtifact)*
    ): Dependencies[Collection] =
      Dependencies(dependency, modules = this.modules ++ modules)

    def toSeq: Seq[ModuleID] = modules.map({ module =>
      val da = module.apply(dependency)
      val newMod = da.resolve(dependency.organization) % dependency.revision
      da.update.fold(newMod)(_.apply(newMod))
    })

    def ++(dependencies: Dependencies[_]): DependencyBuilder =
      DependencyBuilder(
        Seq(this, dependencies)
      )

    def build: Seq[ModuleID] = toSeq
  }

  implicit def appendBuilder[A]: Append.Values[Seq[A], Utils.Builder[Seq[A]]] =
    (a: Seq[A], b: Utils.Builder[Seq[A]]) => a ++ b

  case class DependencyBuilder(
      dependencies: Seq[Dependencies[_]]
  ) extends Utils.Builder[Seq[ModuleID]] {

    def ++(dependencies: Dependencies[_]): DependencyBuilder =
      DependencyBuilder(this.dependencies :+ dependencies)

    def build: Seq[ModuleID] = dependencies.flatMap(_.toSeq)
  }

  def dependencies(moduleIDs: ModuleID*): Utils.Builder[Seq[ModuleID]] = () =>
    moduleIDs

  class DependencyGroup(
      val organization: SBT.Organization,
      val revision: SBT.Version
  ) {

    def single(artifact: Module.Ref): Dependencies[this.type] =
      modules(_ => this.artifact(artifact))

    def withRevision(revision: SBT.Version): DependencyGroup =
      new DependencyGroup(organization = organization, revision = revision)

    private val builder
        : Seq[this.type => DependencyArtifact] => Dependencies[this.type] =
      (modules: Seq[this.type => DependencyArtifact]) =>
        Dependencies[this.type](this, modules)

    def modules(
        modules: (this.type => DependencyArtifact)*
    ): Dependencies[this.type] = builder(modules)

    @inline def artifact(fn: Organization => OrganizationArtifactName): DependencyArtifact = DependencyArtifact(fn)

    @inline def artifact(fn: Module.Ref): DependencyArtifact = fn

    @inline implicit protected def moduleToArtifact(
        module: Module.Ref
    ): DependencyArtifact = DependencyArtifact(_ %% module)
  }

}
