package geekbytes.sbt

import _root_.sbt.Keys._
import _root_.sbt.librarymanagement.ModuleID
import SBT.{Module, Organization, Version}
import _root_.sbt.{Def, Keys, ScmInfo, ThisBuild, _}

trait ProjectUtils {

  def org: Organization
  def projectRef: String

  def scalaVersion: Version
  def crossScalaVersions: Seq[Version]

  def mavenResolvers: Seq[MavenRepository]

  def defaultProjectSettings: Seq[Def.Setting[_]] = Def.settings(
    // To make the default compiler and REPL use Dotty
    Keys.scalaVersion := scalaVersion,
    // To cross compile with Scala 3 and Scala 2
    Keys.crossScalaVersions := (scalaVersion +: crossScalaVersions),
    Keys.organization := org,
    Keys.resolvers ++= mavenResolvers,
    Keys.scalacOptions ++= Seq(
      "-Ylog-classpath"
    )
  )

  implicit class ProjectOps(private val project: Project) {

    @inline def withLibraries(
        dependencies: Utils.Builder[Seq[ModuleID]]*
    ): Project = {
      project.settings(
        libraryDependencies ++= dependencies.flatMap(_.build())
      )
    }

    @inline def withDefaultSettings(): Project = {
      project.settings(
        defaultProjectSettings
      )
    }

  }

  case class externalProject(path: Module.Path) {
    def module(moduleName: Module.Name): ProjectRef = {
      ProjectRef(file(path), moduleName)
    }
    def root = {
      RootProject(file(path))
    }
  }

  def createProject(
      path: Module.Path,
      moduleName: Module.Name,
      moduleRef: Module.Ref
  ): Project = {
    Project(moduleRef, file(path))
      .settings(
        defaultProjectSettings
      )
      .settings(
        Keys.name := moduleName,
        Keys.moduleName := s"$projectRef.$moduleRef"
      )
  }

}

class DefaultProjects(
    val org: Organization,
    val projectRef: String,
    val scalaVersion: Version,
    val crossScalaVersions: Seq[Version] = Nil,
    val mavenResolvers: Seq[MavenRepository] = Nil
) extends ProjectUtils
