package io.foodtechlab.i18n;

import java.util.Locale;
import java.util.Map;

/**
 * Класс которые занимается переводом текстовых констант на указанный язык, учитывая его специфику
 *
 * @author kiryanovvi
 * @since 1.2
 */
public interface I18NTranslator {

    /**
     * Метод должен вернуть true если переводчик может перевести на переданную локаль
     *
     * @param locale локаль на которую должен переводить переводчик
     * @return true если переводчик может перевести на выбранную локаль
     * @since 1.2
     */
    boolean isRightTranslator(Locale locale);

    /**
     * Метод возвращает локаль связанную с текущим переводчиком.
     * Используется для поиска в файлах локализаций (**.properties), файла с нужной локализацией.
     *
     * @return локаль
     */
    Locale getLocale();

    /**
     * Метод возвращает уже переведённую строку, взятую из файла переводов (**.properties).
     * Поиск производится по {@code code}
     *
     * @param code код по которому выполняется поиск локализации
     * @return текст локализации
     * @since 1.2
     */
    String translate(String code);

    /**
     * Метод возвращает уже переведённую строку, взятую из файла переводов (**.properties).
     * Поиск производится по {@code code}.
     * Данный метод поддерживает инъекции.
     * Из HashMap, в возвращаемую строку, будут встраиваться значения хранимые в {@code value}.
     * По {@code key} будет производиться поиск вхождения в текст и будет заменён на {@code value}
     *
     * @param code       code код по которому выполняется поиск локализации
     * @param injections Словарь инъекций. Key - ключ по которому выполняется поиск места инъекции. Value значение встраиваемое в строку
     * @param <T>        для встраивания любого типа используется метод toString()
     * @return текст локализации
     * @since 1.2
     */
    <T> String translate(String code, Map<String, T> injections);
}
