package ru.foodtechlab.lib.auth.integration.proxy.api.credential.v1.controllers;

import com.rcore.domain.commons.port.dto.SearchResult;
import com.rcore.rest.api.commons.response.OkApiResponse;
import com.rcore.rest.api.commons.response.SuccessApiResponse;
import lombok.RequiredArgsConstructor;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;
import ru.foodtechlab.lib.auth.integration.core.credential.CredentialServiceFacade;
import ru.foodtechlab.lib.auth.service.domain.credential.exceptions.CredentialNotFoundException;
import ru.foodtechlab.lib.auth.service.facade.credential.dto.requests.ChangeCredentialPasswordRequest;
import ru.foodtechlab.lib.auth.service.facade.credential.dto.requests.CreateCredentialRequest;
import ru.foodtechlab.lib.auth.service.facade.credential.dto.requests.FindCredentialWithFiltersRequest;
import ru.foodtechlab.lib.auth.service.facade.credential.dto.requests.UpdateCredentialRequest;
import ru.foodtechlab.lib.auth.service.facade.credential.dto.responses.CredentialResponse;
import ru.foodtechlab.lib.auth.service.facade.credential.resources.CredentialResource;
import ru.foodtechlab.lib.auth.service.facade.credential.routers.CredentialRoutes;

@RequiredArgsConstructor
public class CredentialProxyApiController implements CredentialResource {

    private final CredentialServiceFacade credentialServiceFacade;

    @Override
    @PostMapping(value = CredentialRoutes.CHANGE_PASSWORD)
    public SuccessApiResponse<CredentialResponse> changePassword(@PathVariable String id, @RequestBody ChangeCredentialPasswordRequest request) {
        return SuccessApiResponse.of(credentialServiceFacade.changePassword(id, request));
    }

    @Override
    @PostMapping(value = CredentialRoutes.BLOCKED_CHANGE_STATUS)
    public SuccessApiResponse<CredentialResponse> changeStatus(@PathVariable String id) {
        return SuccessApiResponse.of(credentialServiceFacade.changeStatus(id));
    }

    @Override
    @PostMapping(value = CredentialRoutes.ROOT, produces = MediaType.APPLICATION_JSON_VALUE)
    public SuccessApiResponse<CredentialResponse> create(@RequestBody CreateCredentialRequest request) {
        return SuccessApiResponse.of(credentialServiceFacade.create(request));
    }

    @Override
    @GetMapping(value = CredentialRoutes.BY_ID, produces = MediaType.APPLICATION_JSON_VALUE)
    public SuccessApiResponse<CredentialResponse> findById(@PathVariable String id) {
        return SuccessApiResponse.of(credentialServiceFacade.findById(id)
                .orElseThrow(CredentialNotFoundException::new));
    }

    @Override
    @GetMapping(value = CredentialRoutes.BY_NAME)
    public SuccessApiResponse<CredentialResponse> findByName(@RequestParam String name) {
        return SuccessApiResponse.of(credentialServiceFacade.findByName(name)
                .orElseThrow(CredentialNotFoundException::new));
    }

    @Override
    @GetMapping(value = CredentialRoutes.ROOT, produces = MediaType.APPLICATION_JSON_VALUE)
    public SuccessApiResponse<SearchResult<CredentialResponse>> find(@ModelAttribute FindCredentialWithFiltersRequest request) {
        return SuccessApiResponse.of(credentialServiceFacade.find(request));
    }

    @Override
    @PutMapping(value = CredentialRoutes.BY_ID)
    public SuccessApiResponse<CredentialResponse> update(@PathVariable String id, @RequestBody UpdateCredentialRequest updateCredentialRequest) {
        return SuccessApiResponse.of(credentialServiceFacade.update(id, updateCredentialRequest));
    }

    @Override
    @PostMapping(value = CredentialRoutes.BLOCKED_CHANGE_STATUS, produces = MediaType.APPLICATION_JSON_VALUE)
    public SuccessApiResponse<CredentialResponse> changeBlockStatus(@PathVariable String id) {
        return SuccessApiResponse.of(credentialServiceFacade.changeStatus(id));
    }

    @Override
    @DeleteMapping(value = CredentialRoutes.BY_ID, produces = MediaType.APPLICATION_JSON_VALUE)
    public OkApiResponse delete(@PathVariable String id) {
        credentialServiceFacade.delete(id);
        return OkApiResponse.of();
    }


    @Override
    @GetMapping(value = CredentialRoutes.BY_EMAIL, produces = MediaType.APPLICATION_JSON_VALUE)
    public SuccessApiResponse<CredentialResponse> findByEmail(String email) {
        return SuccessApiResponse.of(credentialServiceFacade.findByEmail(email)
                .orElseThrow(CredentialNotFoundException::new));
    }

    @Override
    @GetMapping(value = CredentialRoutes.BY_PHONE_NUMBER, produces = MediaType.APPLICATION_JSON_VALUE)
    public SuccessApiResponse<CredentialResponse> findByPhoneNumber(String phone) {
        return SuccessApiResponse.of(credentialServiceFacade.findByPhoneNumber(phone)
                .orElseThrow(CredentialNotFoundException::new));
    }
}
