/*
 * Decompiled with CFR 0.152.
 */
package io.fliqa.client.interledger.signature;

import io.fliqa.client.interledger.InterledgerClientOptions;
import io.fliqa.client.interledger.InterledgerObjectMapper;
import io.fliqa.client.interledger.exception.InterledgerClientException;
import io.fliqa.client.interledger.utils.Assert;
import java.net.URI;
import java.net.http.HttpRequest;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Signature;
import java.security.SignatureException;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Base64;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

public class SignatureRequestBuilder {
    static final String METHOD = "@method";
    static final String TARGET = "@target-uri";
    static final String SIGNATURE_PARAMS = "@signature-params";
    static final String CONTENT_TYPE_HEADER = "Content-Type";
    static final String CONTENT_DIGEST_HEADER = "Content-Digest";
    static final String CONTENT_LENGTH_HEADER = "Content-Length";
    static final String SIGNATURE_INPUT_HEADER = "Signature-Input";
    static final String SIGNATURE_HEADER = "Signature";
    static final String AUTHORIZATION_HEADER = "Authorization";
    public static final String ACCEPT_HEADER = "Accept";
    public static final String APPLICATION_JSON = "application/json";
    static final String DEFAULT_SIGNATURE_ID = "sig1";
    static final String DIGEST_ALGORITHM = "SHA-512";
    static final String SIGNATURE_ALGORITHM = "Ed25519";
    static final Set<String> ALLOWED_METHODS = Set.of("GET", "POST", "PUT", "DELETE", "HEAD");
    final PrivateKey privateKey;
    final String keyId;
    private long created = 0L;
    final LinkedHashMap<String, Object> parameters = new LinkedHashMap();
    private final InterledgerObjectMapper mapper;
    private String body;

    public SignatureRequestBuilder(PrivateKey privateKey, String keyId, InterledgerObjectMapper mapper) {
        Assert.notNull((Object)privateKey, "PrivateKey cannot be null");
        Assert.notNullOrEmpty(keyId, "KeyId cannot be null or empty");
        this.privateKey = privateKey;
        this.keyId = keyId;
        this.mapper = mapper == null ? new InterledgerObjectMapper() : mapper;
    }

    public SignatureRequestBuilder(PrivateKey privateKey, String keyId) {
        this(privateKey, keyId, null);
    }

    public SignatureRequestBuilder method(String value) {
        Assert.notNullOrEmpty(value, "Method cannot be null or empty!");
        Assert.isTrue(ALLOWED_METHODS.contains(value), String.format("Method '%s' is not allowed. Allowed methods are: %s", value, String.join((CharSequence)", ", ALLOWED_METHODS)));
        this.parameters.put(METHOD, value.toUpperCase());
        return this;
    }

    public SignatureRequestBuilder POST(Object body) {
        this.checkMethod();
        return this.method("POST").json(body);
    }

    public SignatureRequestBuilder POST() {
        this.checkMethod();
        return this.method("POST");
    }

    public SignatureRequestBuilder PUT(Object body) {
        this.checkMethod();
        return this.method("PUT").json(body);
    }

    public SignatureRequestBuilder PUT() {
        this.checkMethod();
        return this.method("PUT");
    }

    public SignatureRequestBuilder GET() {
        this.checkMethod();
        return this.method("GET");
    }

    public SignatureRequestBuilder DELETE() {
        this.checkMethod();
        return this.method("DELETE");
    }

    public String getMethod() {
        this.checkHasParameter(METHOD);
        return this.parameters.get(METHOD).toString();
    }

    public SignatureRequestBuilder target(String value) {
        return this.target(URI.create(value));
    }

    public SignatureRequestBuilder target(URI value) {
        this.parameters.put(TARGET, this.prepareTarget(value));
        return this;
    }

    public SignatureRequestBuilder json(Object object) {
        try {
            String value = this.mapper.writeValueAsString(object);
            return this.json(value);
        }
        catch (InterledgerClientException e) {
            throw new IllegalArgumentException(e.getMessage(), e);
        }
    }

    public SignatureRequestBuilder json(String json) {
        Assert.notNullOrEmpty(json, "JSON cannot be null or empty!");
        this.body = json;
        this.digest(json);
        this.length(json);
        this.parameters.put(CONTENT_TYPE_HEADER, APPLICATION_JSON);
        return this;
    }

    public SignatureRequestBuilder accessToken(String token) {
        Assert.notNullOrEmpty(token, "Token cannot be null or empty!");
        this.parameters.put(AUTHORIZATION_HEADER, this.prepareToken(token));
        return this;
    }

    private String prepareToken(String token) {
        return String.format("GNAP %s", token);
    }

    private void length(String content) {
        int contentLength = content.getBytes(StandardCharsets.UTF_8).length;
        this.parameters.put(CONTENT_LENGTH_HEADER, contentLength);
    }

    private void digest(String value) {
        try {
            String digest = SignatureRequestBuilder.digestContentSha512(value);
            String digestHeader = String.format("%s=:%s:", DIGEST_ALGORITHM.toLowerCase(), digest);
            this.parameters.put(CONTENT_DIGEST_HEADER, digestHeader);
        }
        catch (Exception e) {
            throw new IllegalArgumentException(String.format("Failed to calculate %s digest for: '%s'.", DIGEST_ALGORITHM, value), e);
        }
    }

    void setSignatureParams() {
        Assert.isTrue(this.created > 0L, "Created timestamp must be set before calculating signature params!");
        String signatureParams = this.parameters.keySet().stream().map(item -> "\"" + item.toLowerCase() + "\"").collect(Collectors.joining(" "));
        String params = String.format("(%s);keyid=\"%s\";created=%d", signatureParams, this.keyId, this.created);
        this.parameters.put(SIGNATURE_PARAMS, params);
    }

    public SignatureRequestBuilder build() {
        return this.build(Instant.now().getEpochSecond());
    }

    public SignatureRequestBuilder build(long createdInSeconds) {
        this.created = createdInSeconds;
        this.setSignatureParams();
        return this;
    }

    private String prepareTarget(URI value) {
        Assert.notNull((Object)value, "Target URI cannot be null!");
        String out = value.toString();
        if (!out.endsWith("/") && value.getQuery() == null) {
            return out + "/";
        }
        return out;
    }

    public URI getTarget() {
        this.checkHasParameter(TARGET);
        return URI.create(this.parameters.get(TARGET).toString());
    }

    public String getSignatureParamsHeader() {
        this.checkIsBuild();
        String signatureParams = this.parameters.get(SIGNATURE_PARAMS).toString();
        return String.format("%s=%s", DEFAULT_SIGNATURE_ID, signatureParams);
    }

    private void checkIsBuild() {
        if (this.parameters.get(SIGNATURE_PARAMS) == null) {
            throw new IllegalStateException("Signature must be build before retrieval of signature params header!");
        }
    }

    private void checkHasParameter(String key) {
        Assert.isTrue(this.parameters.containsKey(key), String.format("Parameter '%s' must be set before continuing!", key));
    }

    private void checkMethod() {
        Assert.isFalse(this.parameters.get(METHOD) == null, String.format("Method '%s' already set!", this.parameters.get(METHOD)));
    }

    protected String getSignatureBase() {
        this.checkIsBuild();
        StringBuilder signatureBase = new StringBuilder();
        Iterator<Map.Entry<String, Object>> iterator = this.parameters.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<String, Object> entry = iterator.next();
            signatureBase.append("\"").append(entry.getKey().toLowerCase()).append("\": ").append(entry.getValue());
            if (!iterator.hasNext()) continue;
            signatureBase.append(System.lineSeparator());
        }
        return signatureBase.toString();
    }

    protected static String digestContentSha512(String content) throws NoSuchAlgorithmException {
        Assert.notNullOrEmpty(content, "Content must not be null or empty!");
        MessageDigest digest = MessageDigest.getInstance(DIGEST_ALGORITHM.toUpperCase());
        byte[] bodyBytes = content.getBytes(StandardCharsets.UTF_8);
        byte[] hashedBytes = digest.digest(bodyBytes);
        return Base64.getEncoder().encodeToString(hashedBytes);
    }

    protected String getSignature() {
        try {
            Signature signature = Signature.getInstance(SIGNATURE_ALGORITHM);
            signature.initSign(this.privateKey);
            signature.update(this.getSignatureBase().getBytes(StandardCharsets.UTF_8));
            byte[] signatureBytes = signature.sign();
            return Base64.getEncoder().encodeToString(signatureBytes);
        }
        catch (InvalidKeyException | NoSuchAlgorithmException | SignatureException ex) {
            throw new IllegalStateException("Failed to generate signature!", ex);
        }
    }

    String getSignatureHeader() {
        return String.format("%s=:%s:", DEFAULT_SIGNATURE_ID, this.getSignature());
    }

    public LinkedHashMap<String, String> getHeaders() {
        this.checkIsBuild();
        LinkedHashMap<String, String> headers = new LinkedHashMap<String, String>();
        headers.put(ACCEPT_HEADER, APPLICATION_JSON);
        if (this.parameters.containsKey(CONTENT_LENGTH_HEADER)) {
            headers.put(CONTENT_TYPE_HEADER, this.parameters.get(CONTENT_TYPE_HEADER).toString());
        }
        if (this.parameters.containsKey(CONTENT_DIGEST_HEADER)) {
            headers.put(CONTENT_DIGEST_HEADER, this.parameters.get(CONTENT_DIGEST_HEADER).toString());
        }
        if (this.parameters.containsKey(AUTHORIZATION_HEADER)) {
            headers.put(AUTHORIZATION_HEADER, this.parameters.get(AUTHORIZATION_HEADER).toString());
        }
        headers.put(SIGNATURE_INPUT_HEADER, this.getSignatureParamsHeader());
        headers.put(SIGNATURE_HEADER, this.getSignatureHeader());
        return headers;
    }

    public String getBody() {
        return this.body;
    }

    public HttpRequest getRequest(InterledgerClientOptions options) {
        return this.getBuilder(options).build();
    }

    public HttpRequest.Builder getBuilder(InterledgerClientOptions options) {
        if (this.parameters.get(SIGNATURE_PARAMS) == null) {
            this.build();
        }
        this.checkIsBuild();
        HttpRequest.Builder builder = HttpRequest.newBuilder(this.getTarget());
        for (Map.Entry<String, String> entry : this.getHeaders().entrySet()) {
            builder.header(entry.getKey(), entry.getValue());
        }
        if (this.body == null || this.body.isBlank()) {
            builder.method(this.getMethod(), HttpRequest.BodyPublishers.noBody());
        } else {
            builder.method(this.getMethod(), HttpRequest.BodyPublishers.ofString(this.body, StandardCharsets.UTF_8));
        }
        builder.timeout(Duration.of(options.timeOutInSeconds, ChronoUnit.SECONDS));
        return builder;
    }
}

