/*
 * Decompiled with CFR 0.152.
 */
package io.fliqa.client.interledger;

import io.fliqa.client.interledger.InterledgerApiClient;
import io.fliqa.client.interledger.InterledgerClientOptions;
import io.fliqa.client.interledger.InterledgerObjectMapper;
import io.fliqa.client.interledger.exception.InterledgerClientException;
import io.fliqa.client.interledger.logging.HttpLogger;
import io.fliqa.client.interledger.model.AccessAction;
import io.fliqa.client.interledger.model.AccessGrant;
import io.fliqa.client.interledger.model.AccessItemType;
import io.fliqa.client.interledger.model.ApiError;
import io.fliqa.client.interledger.model.GrantAccessRequest;
import io.fliqa.client.interledger.model.IncomingPayment;
import io.fliqa.client.interledger.model.InteractRef;
import io.fliqa.client.interledger.model.OutgoingPayment;
import io.fliqa.client.interledger.model.OutgoingPaymentRequest;
import io.fliqa.client.interledger.model.Payment;
import io.fliqa.client.interledger.model.PaymentPointer;
import io.fliqa.client.interledger.model.PaymentRequest;
import io.fliqa.client.interledger.model.Quote;
import io.fliqa.client.interledger.model.QuoteRequest;
import io.fliqa.client.interledger.model.WalletAddress;
import io.fliqa.client.interledger.signature.SignatureRequestBuilder;
import io.fliqa.client.interledger.utils.Assert;
import java.io.IOException;
import java.math.BigDecimal;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.security.PrivateKey;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class InterledgerApiClientImpl
implements InterledgerApiClient {
    private static final Logger LOGGER = LoggerFactory.getLogger(InterledgerApiClientImpl.class);
    private static final String ILP_METHOD = "ilp";
    private final WalletAddress clientWallet;
    private final PrivateKey privateKey;
    private final String keyId;
    private final HttpClient client;
    private final InterledgerClientOptions options;
    private final InterledgerObjectMapper mapper = new InterledgerObjectMapper();
    private final HttpLogger httpLogger;

    public InterledgerApiClientImpl(WalletAddress clientWallet, PrivateKey privateKey, String keyId, InterledgerClientOptions options) {
        Assert.notNull((Object)clientWallet, "WalletAddress cannot be null");
        Assert.notNull((Object)privateKey, "PrivateKey cannot be null");
        Assert.notNullOrEmpty(keyId, "KeyId cannot be null or empty");
        Assert.notNull((Object)options, "InterledgerClientOptions cannot be null");
        this.clientWallet = clientWallet;
        this.privateKey = privateKey;
        this.keyId = keyId;
        this.options = options;
        this.client = InterledgerApiClientImpl.createDefaultHttpClient(options);
        this.httpLogger = new HttpLogger(LOGGER);
    }

    public InterledgerApiClientImpl(WalletAddress clientWallet, PrivateKey privateKey, String keyId) {
        this(clientWallet, privateKey, keyId, InterledgerClientOptions.DEFAULT);
    }

    protected static HttpClient createDefaultHttpClient(InterledgerClientOptions options) {
        return HttpClient.newBuilder().connectTimeout(Duration.ofSeconds(options.connectTimeOutInSeconds)).build();
    }

    @Override
    public PaymentPointer getWallet(WalletAddress address) throws InterledgerClientException {
        Assert.notNull((Object)address, "WalletAddress cannot be null");
        LOGGER.debug("getWallet: {}", (Object)address);
        HttpRequest request = HttpRequest.newBuilder(address.paymentPointer).GET().header("Accept".toLowerCase(), "application/json").timeout(Duration.of(this.options.timeOutInSeconds, ChronoUnit.SECONDS)).build();
        return this.send(request, PaymentPointer.class);
    }

    @Override
    public AccessGrant createPendingGrant(PaymentPointer receiver) throws InterledgerClientException {
        Assert.notNull((Object)receiver, "PaymentPointer receiver cannot be null");
        LOGGER.debug("createPendingGrant: {}", (Object)receiver);
        GrantAccessRequest accessRequest = GrantAccessRequest.build(this.clientWallet, AccessItemType.incomingPayment, Set.of(AccessAction.read, AccessAction.complete, AccessAction.create));
        HttpRequest request = new SignatureRequestBuilder(this.privateKey, this.keyId, this.mapper).POST(accessRequest).target(receiver.authServer).getRequest(this.options);
        return this.send(request, AccessGrant.class);
    }

    @Override
    public IncomingPayment createIncomingPayment(PaymentPointer receiver, AccessGrant pendingGrant, BigDecimal amount) throws InterledgerClientException {
        Assert.notNull((Object)receiver, "PaymentPointer receiver cannot be null");
        Assert.notNull((Object)pendingGrant, "AccessGrant pendingGrant cannot be null");
        Assert.notNull((Object)amount, "BigDecimal amount cannot be null");
        LOGGER.debug("createIncomingPayment: {} for: {}", (Object)receiver, (Object)amount);
        PaymentRequest paymentRequest = PaymentRequest.build(receiver, amount, this.options.transactionExpirationInSeconds);
        HttpRequest request = new SignatureRequestBuilder(this.privateKey, this.keyId, this.mapper).POST(paymentRequest).target(this.buildResourceUrl(receiver.resourceServer, "/incoming-payments")).accessToken(this.extractAccessToken(pendingGrant)).getRequest(this.options);
        return this.send(request, IncomingPayment.class);
    }

    @Override
    public AccessGrant createQuoteRequest(PaymentPointer sender) throws InterledgerClientException {
        Assert.notNull((Object)sender, "PaymentPointer sender cannot be null");
        LOGGER.debug("createQuoteRequest: {}", (Object)sender);
        GrantAccessRequest accessRequest = GrantAccessRequest.build(this.clientWallet, AccessItemType.quote, Set.of(AccessAction.read, AccessAction.create));
        HttpRequest request = new SignatureRequestBuilder(this.privateKey, this.keyId, this.mapper).POST(accessRequest).target(sender.authServer).getRequest(this.options);
        return this.send(request, AccessGrant.class);
    }

    @Override
    public Quote createQuote(String quoteToken, PaymentPointer sender, IncomingPayment incomingPayment) throws InterledgerClientException {
        Assert.notNullOrEmpty(quoteToken, "Quote token cannot be null or empty");
        Assert.notNull((Object)sender, "PaymentPointer sender cannot be null");
        Assert.notNull((Object)incomingPayment, "IncomingPayment cannot be null");
        LOGGER.debug("createQuote: {} for: {}", (Object)incomingPayment, (Object)sender);
        QuoteRequest quoteRequest = QuoteRequest.build(sender.address, incomingPayment.id.toString(), ILP_METHOD);
        HttpRequest request = new SignatureRequestBuilder(this.privateKey, this.keyId, this.mapper).POST(quoteRequest).target(this.buildResourceUrl(sender.resourceServer, "/quotes")).accessToken(quoteToken).getRequest(this.options);
        return this.send(request, Quote.class);
    }

    @Override
    public OutgoingPayment continueGrant(PaymentPointer sender, Quote quote, URI returnUrl, String nonce) throws InterledgerClientException {
        Assert.notNull((Object)sender, "PaymentPointer sender cannot be null");
        Assert.notNull((Object)quote, "Quote cannot be null");
        Assert.notNull((Object)returnUrl, "Return URL cannot be null");
        Assert.notNullOrEmpty(nonce, "Nonce cannot be null or empty");
        LOGGER.debug("continueGrant: {} for: {}", (Object)quote, (Object)sender);
        GrantAccessRequest accessRequest = GrantAccessRequest.outgoing(this.clientWallet, AccessItemType.outgoingPayment, Set.of(AccessAction.read, AccessAction.create), sender.address, quote.debitAmount).redirectInteract(returnUrl, nonce);
        HttpRequest request = new SignatureRequestBuilder(this.privateKey, this.keyId, this.mapper).POST(accessRequest).target(sender.authServer).getRequest(this.options);
        return this.send(request, OutgoingPayment.class);
    }

    @Override
    public AccessGrant finalizeGrant(OutgoingPayment outgoingPayment, String interactRef) throws InterledgerClientException {
        Assert.notNull((Object)outgoingPayment, "OutgoingPayment cannot be null");
        Assert.notNullOrEmpty(interactRef, "Interact reference cannot be null or empty");
        LOGGER.debug("finalizeGrant: {} for: {}", (Object)outgoingPayment, (Object)interactRef);
        InteractRef ref = InteractRef.build(interactRef);
        HttpRequest request = new SignatureRequestBuilder(this.privateKey, this.keyId, this.mapper).POST(ref).target(outgoingPayment.paymentContinue.uri).accessToken(this.extractContinueAccessToken(outgoingPayment)).getRequest(this.options);
        return this.send(request, AccessGrant.class);
    }

    @Override
    public Payment finalizePayment(AccessGrant finalizedGrant, PaymentPointer senderWallet, Quote quote) throws InterledgerClientException {
        Assert.notNull((Object)finalizedGrant, "AccessGrant finalizedGrant cannot be null");
        Assert.notNull((Object)senderWallet, "PaymentPointer senderWallet cannot be null");
        Assert.notNull((Object)quote, "Quote cannot be null");
        LOGGER.debug("finalizePayment: {} for: {}, with: {}", new Object[]{finalizedGrant, senderWallet, quote});
        OutgoingPaymentRequest outgoingPayment = new OutgoingPaymentRequest();
        outgoingPayment.quoteId = quote.id;
        outgoingPayment.walletAddress = senderWallet.address;
        HttpRequest request = new SignatureRequestBuilder(this.privateKey, this.keyId, this.mapper).POST(outgoingPayment).target(this.buildResourceUrl(senderWallet.resourceServer, "/outgoing-payments")).accessToken(this.extractAccessToken(finalizedGrant)).getRequest(this.options);
        return this.send(request, Payment.class);
    }

    @Override
    public IncomingPayment getIncomingPayment(IncomingPayment payment, AccessGrant grant) throws InterledgerClientException {
        Assert.notNull((Object)payment, "IncomingPayment cannot be null");
        Assert.notNull((Object)grant, "AccessGrant cannot be null");
        LOGGER.debug("getIncomingPayment: {}", (Object)payment);
        HttpRequest request = new SignatureRequestBuilder(this.privateKey, this.keyId, this.mapper).GET().target(payment.id).accessToken(this.extractAccessToken(grant)).getRequest(this.options);
        return this.send(request, IncomingPayment.class);
    }

    private String extractAccessToken(AccessGrant grant) {
        Assert.notNull((Object)grant, "AccessGrant cannot be null");
        Assert.notNull((Object)grant.access, "AccessGrant.access cannot be null");
        Assert.notNull((Object)grant.access.token, "AccessGrant.access.token cannot be null");
        return grant.access.token;
    }

    private String extractContinueAccessToken(OutgoingPayment outgoingPayment) {
        Assert.notNull((Object)outgoingPayment, "OutgoingPayment cannot be null");
        Assert.notNull((Object)outgoingPayment.paymentContinue, "OutgoingPayment.paymentContinue cannot be null");
        Assert.notNull((Object)outgoingPayment.paymentContinue.access, "OutgoingPayment.paymentContinue.access cannot be null");
        Assert.notNull((Object)outgoingPayment.paymentContinue.access.token, "OutgoingPayment.paymentContinue.access.token cannot be null");
        return outgoingPayment.paymentContinue.access.token;
    }

    private URI buildResourceUrl(URI baseUri, String path) {
        Assert.notNull((Object)baseUri, "Base URI cannot be null");
        Assert.notNull((Object)path, "Path cannot be null");
        Object normalizedPath = path.startsWith("/") ? path : "/" + path;
        return baseUri.resolve((String)normalizedPath);
    }

    public <T> T send(HttpRequest request, Class<T> responseType) throws InterledgerClientException {
        try {
            this.httpLogger.logRequest(request);
            HttpResponse<String> response = this.client.send(request, HttpResponse.BodyHandlers.ofString());
            this.httpLogger.logResponse(response);
            if (response.statusCode() < 200 || response.statusCode() >= 300) {
                this.handleHttpError(response);
            }
            return this.mapper.readValue(response.body(), responseType);
        }
        catch (IOException e) {
            throw new InterledgerClientException(e);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterledgerClientException(e);
        }
    }

    private void handleHttpError(HttpResponse<String> response) throws InterledgerClientException, IOException {
        int statusCode = response.statusCode();
        ApiError error = this.mapper.readError(response.body(), statusCode);
        if (statusCode >= 400 && statusCode < 500) {
            LOGGER.warn("Client error [{}]: {}", (Object)statusCode, (Object)error.description);
        } else if (statusCode >= 500 && statusCode < 600) {
            LOGGER.error("Server error [{}]: {}", (Object)statusCode, (Object)error.description);
        } else {
            LOGGER.warn("Unexpected HTTP status [{}]: {}", (Object)statusCode, (Object)error.description);
        }
        throw InterledgerClientException.getApiException(error, response);
    }
}

