package eventcenter.leveldb;

import eventcenter.api.CommonEventSource;
import eventcenter.api.EventListener;
import eventcenter.api.async.EventQueue;
import eventcenter.api.async.LockBase;
import eventcenter.api.async.MessageListener;
import eventcenter.api.async.QueueException;
import eventcenter.api.tx.EventTxnStatus;
import eventcenter.api.tx.ResumeTxnHandler;
import eventcenter.api.tx.TransactionalSupport;
import eventcenter.leveldb.strategy.LimitReadHouseKeepingStrategy;
import eventcenter.leveldb.tx.TransactionConfig;
import eventcenter.leveldb.tx.TxnQueueComponent;
import eventcenter.leveldb.tx.TxnQueueComponentFactory;
import eventcenter.leveldb.tx.UnopenTxnModeException;
import org.apache.log4j.Logger;
import org.iq80.leveldb.WriteBatch;

import java.io.IOException;

/**
 * use read cursor and write cursor to offer and transfer data
 * @author JackyLIU
 *
 */
public class LevelDBQueue extends LockBase implements EventQueue, TransactionalSupport {

	private final QueueMiddleComponent adapter;
	
	private volatile boolean isOpen = false;
	
	private final Logger logger = Logger.getLogger(this.getClass());
	
	private HouseKeepingStrategy houseKeepingStrategy;

	/**
	 * 是否支持事务，设置了这个类表示需要开启事务模式
	 */
	private TransactionConfig transactionConfig;

	/**
	 * 当设置了 {@link #transactionConfig}之后，将会在 {@link #open()}的时候开启此控件
	 */
	private TxnQueueComponent txnQueueComponent;

	/**
	 * 事务容量数，设置了{@link #txnQueueComponent}才有效
	 */
	private Integer txnCapacity = 1000;
	
	public LevelDBQueue(QueueMiddleComponent adapter){
		this.adapter = adapter;
	}
	
	@Override
	public void open() throws IOException{
		final long start = System.currentTimeMillis();
		isOpen = true;
		this.adapter.open();
		openTxnQueueComponent();
		if(null == houseKeepingStrategy){
			// default create Limit ReadHouseKeeping
			houseKeepingStrategy = new LimitReadHouseKeepingStrategy(this);
		}
		houseKeepingStrategy.open();
		logger.info(String.format("startup leveldb queue[%s,%s] success, took: %s ms.", adapter.getQueueName(), this.enqueueSize(), (System.currentTimeMillis() - start)));
	}

	public boolean isOpenTxnMode(){
		return null != transactionConfig;
	}

	private void openTxnQueueComponent(){
		if(!isOpenTxnMode()) {
			return ;
		}
		txnQueueComponent = TxnQueueComponentFactory.create(adapter.getQueueName(), adapter.getDB(), transactionConfig, txnCapacity);
		try {
			txnQueueComponent.open();
			adapter.setOpenTxn(true);
		} catch (Exception e) {
			logger.error("startup txnQueueComponent failure:" + e.getMessage(), e);
		}
	}
	
	@Override
	public void close() throws IOException {
		final long start = System.currentTimeMillis();
		if(null != txnQueueComponent){
			txnQueueComponent.shutdown();
		}
		this.adapter.close();
		this.houseKeepingStrategy.close();
		isOpen = false;
		synchronized(this){
			this.notifyAll();
		}
		logger.info(String.format("close leveldb queue[%s] success, took: %s ms.", adapter.getQueueName(), (System.currentTimeMillis() - start)));
	}

	@Override
	public void offer(CommonEventSource evt) throws QueueException {
		try {
			adapter.save(evt);
			unlock();
		} catch (PersistenceException e) {
			throw new QueueException(e);
		}
	}

	@Override
	public void offer(CommonEventSource evt, long timeout) {
		offer(evt);
	}

	@Override
	public CommonEventSource peek() {
		try {
			EventSourceWrapper event = adapter.peek();
			if(null == event){
				return event;
			}
			return event.getEvt();
		} catch (PersistenceException e) {
			throw new QueueException(e);
		}
	}

	@Override
	public CommonEventSource peek(long timeout) {
		if(!isOpen){
			return null;
		}

		CommonEventSource evt = peek();
		if(evt != null){
			return evt;
		}

		lock(timeout);
		if(timeout <= -1){
			return peek(timeout);
		}

		return peek();
	}

	@Override
	public CommonEventSource transfer() {
		return transfer(-1L);
	}

	@Override
	public CommonEventSource transfer(long timeout) {
		if(!isOpen){
			return null;
		}
		
		CommonEventSource evt = pop();
		if(evt != null) {
			return evt;
		}

		lock(timeout);
		if(timeout <= 0) {
			return transfer(timeout);
		}

		return pop();
	}

	CommonEventSource transfer(long timeout, WriteBatch wb){
		if(!isOpen){
			return null;
		}

		CommonEventSource evt = pop(wb);
		if(evt != null) {
			return evt;
		}

		lock(timeout);
		if(timeout < 0) {
			return transfer(timeout, wb);
		}

		return pop(wb);
	}

	private CommonEventSource pop(){
		if(!isOpen) {
			return null;
		}
		try {
			EventSourceWrapper eventWrapper = adapter.pop();
			if(null == eventWrapper) {
				return null;
			}
			if(!isOpenTxnMode()) {
				return eventWrapper.getWrapper();
			}
			return eventWrapper;
		} catch (Throwable e) {
			logger.error("pop message error", e);
			return null;
		}
	}

	private CommonEventSource pop(WriteBatch wb){
		if(!isOpen) {
			return null;
		}
		try {
			EventSourceWrapper eventWrapper = adapter.pop(wb);
			if(null == eventWrapper) {
				return null;
			}
			if(!isOpenTxnMode()) {
				return eventWrapper.getEvt();
			}
			return eventWrapper;
		} catch (Throwable e) {
			logger.error("pop message error", e);
			return null;
		}
	}

	@Override
	public int enqueueSize() {
		return (int)adapter.count();
	}

	@Override
	public boolean isEmpty() {
		return false;
	}

	@Override
	public void setMessageListener(MessageListener listener) {

	}

	@Override
	public void resumeTxn(ResumeTxnHandler handler) throws Exception {
		getTxnQueueComponent().resumeTxn(handler);
	}

	@Override
	public EventTxnStatus getTxnStatus(CommonEventSource event, String txnId, EventListener listener) throws Exception {
		if(!isOpenTxnMode()) {
			throw new UnopenTxnModeException();
		}
		return getTxnQueueComponent().getTxnStatus(event.getEventId(), listener.getClass(), txnId);
	}

	public EventTxnStatus getTxnStatus(CommonEventSource event, String txnId, EventListener listener, WriteBatch writeBatch) throws Exception {
		if(!isOpenTxnMode()) {
			throw new UnopenTxnModeException();
		}
		return getTxnQueueComponent().getTxnStatus(event.getEventId(), listener.getClass(), txnId, writeBatch);
	}

	@Override
	public void commit(EventTxnStatus txnStatus) throws Exception {
		if(!isOpenTxnMode()) {
			throw new UnopenTxnModeException();
		}

		getTxnQueueComponent().commit(txnStatus);
	}

	public void houseKeeping() throws PersistenceException, IOException {
		getQueueMiddleComponent().houseKeeping();
		if(isOpenTxnMode()){
			getTxnQueueComponent().houseKeeping();
		}
	}

	public HouseKeepingStrategy getHouseKeepingStrategy() {
		return houseKeepingStrategy;
	}

	public void setHouseKeepingStrategy(HouseKeepingStrategy houseKeepingStrategy) {
		this.houseKeepingStrategy = houseKeepingStrategy;
	}

	public TransactionConfig getTransactionConfig() {
		return transactionConfig;
	}

	public void setTransactionConfig(TransactionConfig transactionConfig) {
		this.transactionConfig = transactionConfig;
	}

	public Integer getTxnCapacity() {
		return txnCapacity;
	}

	public void setTxnCapacity(Integer txnCapacity) {
		this.txnCapacity = txnCapacity;
	}

	public TxnQueueComponent getTxnQueueComponent() {
		return txnQueueComponent;
	}

	public QueueMiddleComponent getQueueMiddleComponent(){
		return adapter;
	}
}
