package io.resys.thena.client.sample.spi;

import io.resys.thena.client.sample.BatchConsumerTable;
import io.resys.thena.client.sample.FindAllFilter;
import io.resys.thena.client.sample.FindAllUsingSqlBuilder;
import io.resys.thena.client.sample.entities.BatchConsumer;
import io.resys.thena.datasource.ImmutableSql;
import io.resys.thena.datasource.ImmutableSqlTuple;
import io.resys.thena.datasource.ImmutableSqlTupleList;
import io.resys.thena.datasource.ThenaSqlClient;
import io.resys.thena.datasource.ThenaSqlDataSource;
import io.resys.thena.datasource.ThenaSqlDataSourceErrorHandler;
import io.vertx.mutiny.sqlclient.Tuple;
import java.lang.Override;
import java.lang.String;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;
import lombok.AllArgsConstructor;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Value
@AllArgsConstructor
public final class BatchConsumerTableImpl implements BatchConsumerTable {
  Batch2TableNames tables;

  ThenaSqlDataSource dataSource;

  ThenaSqlDataSourceErrorHandler errorHandler;

  public BatchConsumerTableImpl(Batch2TableNames tables, ThenaSqlDataSource dataSource) {
    this.tables = tables;
    this.dataSource = dataSource;
    this.errorHandler = dataSource.getErrorHandler();
  }

  @Override
  public ThenaSqlClient.Sql findAll() {
    return ImmutableSql.builder().value("SELECT * FROM " + tables.getBatchConsumers() + "").rowMapper(new BatchConsumerTable.BatchConsumerMapper()).build();
  }

  @Override
  public ThenaSqlClient.SqlTuple getById(String id) {
    return ImmutableSqlTuple.builder().value("SELECT * FROM " + tables.getBatchConsumers() + " WHERE id = $1").props(Tuple.of(id)).rowMapper(new BatchConsumerTable.BatchConsumerMapper()).build();
  }

  @Override
  public ThenaSqlClient.Sql findAllEnabled() {
    return ImmutableSql.builder().value("SELECT * FROM " + tables.getBatchConsumers() + " WHERE consumer_status = 'ENABLED'").rowMapper(new BatchConsumerTable.BatchConsumerMapper()).build();
  }

  @Override
  public ThenaSqlClient.SqlTuple findAllEnabledByAppId(String appId) {
    return ImmutableSqlTuple.builder().value("SELECT * FROM " + tables.getBatchConsumers() + " WHERE consumer_status = 'ENABLED' AND app_id = $1").props(Tuple.of(appId)).rowMapper(new BatchConsumerTable.BatchConsumerMapper()).build();
  }

  @Override
  public ThenaSqlClient.SqlTuple findAllUsingSqlBuilder(FindAllFilter filter) {
    final var sqlBuilder = new FindAllUsingSqlBuilder();
    final var baseSql = sqlBuilder.apply(dataSource.getTenant(), "", filter);

    var sqlValue = baseSql.getValue();
    sqlValue = sqlValue.replaceAll("(?i)\\{batch_consumers\\}", tables.getBatchConsumers());

    return ImmutableSqlTuple.builder().value(sqlValue).props(baseSql.getProps()).rowMapper(new BatchConsumerTable.BatchConsumerMapper()).build();
  }

  @Override
  public ThenaSqlClient.SqlTupleList insertMany(List<BatchConsumer> users) {
    final var mapper = new BatchConsumerTable.BatchConsumerInsertMapper();
    return ImmutableSqlTupleList.builder()
          .value("INSERT INTO " + tables.getBatchConsumers() + " (id, app_id, batch_name, consumer_name, consumer_qualified_java_name, consumer_status, consumer_comment, consumer_created_at, consumer_created_by, consumer_updated_at, consumer_updated_by) VALUES($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11)")
          .props(users.stream()
            .map(mapper::apply)
            .collect(Collectors.toList()))
          .build();
  }

  @Override
  public ThenaSqlClient.SqlTupleList updateMany(List<BatchConsumer> users) {
    final var mapper = new BatchConsumerTable.BatchConsumerUpdateMapper();
    return ImmutableSqlTupleList.builder()
          .value("UPDATE " + tables.getBatchConsumers() + " SET consumer_qualified_java_name = $1, consumer_status = $2, batch_name = $3, consumer_updated_at = $4, consumer_updated_by = $5, consumer_comment = $6 WHERE id = $7")
          .props(users.stream()
            .map(mapper::apply)
            .collect(Collectors.toList()))
          .build();
  }

  @Override
  public ThenaSqlClient.SqlTuple findAllByAppId(String appId, boolean lockForUpdate) {
    return ImmutableSqlTuple.builder().value("SELECT consumers.* FROM " + tables.getBatchConsumers() + " AS consumers WHERE consumers.app_id = $1 FOR UPDATE").props(Tuple.of(appId, lockForUpdate)).rowMapper(new BatchConsumerTable.BatchConsumerMapper()).build();
  }

  @Override
  public ThenaSqlClient.SqlTuple deleteById(String id) {
    return ImmutableSqlTuple.builder()
          .value("DELETE FROM " + tables.getBatchConsumers() + " WHERE id = $1")
          .props(new BatchConsumerTable.BatchConsumerIdMapper().apply(id))
          .build();
  }

  @Override
  public ThenaSqlClient.SqlTupleList deleteAll(Collection<BatchConsumer> missionId) {
    final var mapper = new BatchConsumerTable.BatchConsumerDeleteMapper();
    return ImmutableSqlTupleList.builder()
          .value("DELETE FROM " + tables.getBatchConsumers() + " WHERE id = $1")
          .props(missionId.stream()
            .map(mapper::apply)
            .collect(Collectors.toList()))
          .build();
  }

  public ThenaSqlClient.Sql createTable() {
    return ImmutableSql.builder().value("CREATE TABLE IF NOT EXISTS " + tables.getBatchConsumers() + " ( id VARCHAR(40) PRIMARY KEY, batch_name TEXT NOT NULL, app_id TEXT NOT NULL, consumer_name TEXT NOT NULL, consumer_qualified_java_name TEXT NOT NULL, consumer_comment TEXT NOT NULL, consumer_status VARCHAR(100) NOT NULL, consumer_created_at TIMESTAMP WITH TIME ZONE NOT NULL, consumer_created_by TEXT NOT NULL, consumer_updated_at TIMESTAMP WITH TIME ZONE, consumer_updated_by TEXT, UNIQUE(batch_name, app_id, consumer_name) ); CREATE INDEX IF NOT EXISTS " + tables.getBatchConsumers() + "_APP_INDEX ON " + tables.getBatchConsumers() + " (app_id); CREATE INDEX IF NOT EXISTS " + tables.getBatchConsumers() + "_BATCH_NAME_INDEX ON " + tables.getBatchConsumers() + " (batch_name); CREATE INDEX IF NOT EXISTS " + tables.getBatchConsumers() + "_NAME_INDEX ON " + tables.getBatchConsumers() + " (consumer_name);").build();
  }

  public ThenaSqlClient.Sql createConstraints() {
    return ImmutableSql.builder().value("--- constraints for " + tables.getBatchConsumers() + "").build();
  }

  public ThenaSqlClient.Sql dropTable() {
    return ImmutableSql.builder().value("DROP TABLE " + tables.getBatchConsumers() + ";").build();
  }
}
