package io.resys.thena.client.sample.spi;

import io.resys.thena.api.LogConstants;
import io.resys.thena.api.annotations.TenantSql;
import io.resys.thena.client.sample.Batch2DbQuery;
import io.resys.thena.client.sample.FindAllFilter;
import io.resys.thena.client.sample.ImmutableWorld;
import io.resys.thena.client.sample.entities.BatchConsumer;
import io.resys.thena.datasource.ThenaSqlDataSource;
import io.resys.thena.datasource.ThenaSqlDataSourceErrorHandler;
import io.smallrye.mutiny.Multi;
import io.smallrye.mutiny.Uni;
import io.vertx.mutiny.sqlclient.RowSet;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.List;
import java.util.stream.Collectors;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

@Slf4j(
    topic = LogConstants.SHOW_SQL
)
@RequiredArgsConstructor
public class Batch2DbQueryImpl implements Batch2DbQuery {
  private final ThenaSqlDataSource dataSource;

  private final Batch2Registry registry;

  private final ThenaSqlDataSourceErrorHandler errorHandler;

  public Batch2DbQueryImpl(ThenaSqlDataSource dataSource) {
    this.dataSource = dataSource;
    final var names = Batch2TableNames.defaults().toRepo(dataSource.getTenant());
    this.registry = new Batch2Registry(names, dataSource);
    this.errorHandler = dataSource.getErrorHandler();
  }

  @Override
  public Uni<World> findAll() {
    return Uni.combine().all()
      .unis(
        queryBatchConsumers().findAll()
      )
    .with(sets -> {
      final var builder = ImmutableWorld.builder();
      final List<BatchConsumer> item_0 = (List<BatchConsumer>) sets.get(0);

      builder.batchConsumers(item_0
        .stream()
        .collect(Collectors.toMap(
          e -> e.getId(),
          e -> e
        )));

      return builder.build();
    });
  }

  @Override
  public BatchConsumersQuery queryBatchConsumers() {
    return new BatchConsumersQuery() {

      @Override
      public Uni<List<BatchConsumer>> findAll() {
        final var sql = registry.batchConsumers().findAll();

        if(log.isDebugEnabled()) {
          log.debug("Batch2DbQueryImpl.queryBatchConsumers.findAll query, with props: {} \r\n{}",
            "",
            sql.getValue());
        }

        @SuppressWarnings("unchecked")
        final TenantSql.RowMapper<BatchConsumer> mapper = (TenantSql.RowMapper<BatchConsumer>) sql.getRowMapper();

        return dataSource.getClient().preparedQuery(sql.getValue())
          .mapping(mapper::apply)
          .execute()
          .onItem()
          .transformToUni((RowSet<BatchConsumer> rowset) -> Multi.createFrom().iterable(rowset).collect().asList())
          .onFailure().invoke(e -> errorHandler.deadEnd(new ThenaSqlDataSourceErrorHandler.SqlFailed("Can't find 'BATCH_CONSUMERS's!", sql, e)));
      }

      @Override
      public Uni<BatchConsumer> getById(String id) {
        final var sql = registry.batchConsumers().getById(id);

        if(log.isDebugEnabled()) {
          log.debug("Batch2DbQueryImpl.queryBatchConsumers.getById query, with props: {} \r\n{}",
            sql.getProps().deepToString(),
            sql.getValue());
        }

        @SuppressWarnings("unchecked")
        final TenantSql.RowMapper<BatchConsumer> mapper = (TenantSql.RowMapper<BatchConsumer>) sql.getRowMapper();

        return dataSource.getClient().preparedQuery(sql.getValue())
          .mapping(mapper::apply)
          .execute(sql.getProps())
          .onItem()
          .transform((RowSet<BatchConsumer> rowset) -> {
            final var iterator = rowset.iterator();
            if(iterator.hasNext()) {
              return iterator.next();
            }
            throw new Batch2FindException("io.resys.thena.client.sample.entities.BatchConsumer not found!");
          })
          .onFailure().invoke(e -> errorHandler.deadEnd(new ThenaSqlDataSourceErrorHandler.SqlTupleFailed("Can't find 'BATCH_CONSUMERS's!", sql, e)));
      }

      @Override
      public Multi<BatchConsumer> findAllEnabled() {
        final var sql = registry.batchConsumers().findAllEnabled();

        if(log.isDebugEnabled()) {
          log.debug("Batch2DbQueryImpl.queryBatchConsumers.findAllEnabled query, with props: {} \r\n{}",
            "",
            sql.getValue());
        }

        @SuppressWarnings("unchecked")
        final TenantSql.RowMapper<BatchConsumer> mapper = (TenantSql.RowMapper<BatchConsumer>) sql.getRowMapper();

        return dataSource.getClient().preparedQuery(sql.getValue())
          .mapping(mapper::apply)
          .execute()
          .onItem()
          .transformToMulti((RowSet<BatchConsumer> rowset) -> Multi.createFrom().iterable(rowset))
          .onFailure().invoke(e -> errorHandler.deadEnd(new ThenaSqlDataSourceErrorHandler.SqlFailed("Can't find 'BATCH_CONSUMERS's!", sql, e)));
      }

      @Override
      public Uni<List<BatchConsumer>> findAllEnabledByAppId(String appId) {
        final var sql = registry.batchConsumers().findAllEnabledByAppId(appId);

        if(log.isDebugEnabled()) {
          log.debug("Batch2DbQueryImpl.queryBatchConsumers.findAllEnabledByAppId query, with props: {} \r\n{}",
            sql.getProps().deepToString(),
            sql.getValue());
        }

        @SuppressWarnings("unchecked")
        final TenantSql.RowMapper<BatchConsumer> mapper = (TenantSql.RowMapper<BatchConsumer>) sql.getRowMapper();

        return dataSource.getClient().preparedQuery(sql.getValue())
          .mapping(mapper::apply)
          .execute(sql.getProps())
          .onItem()
          .transformToUni((RowSet<BatchConsumer> rowset) -> Multi.createFrom().iterable(rowset).collect().asList())
          .onFailure().invoke(e -> errorHandler.deadEnd(new ThenaSqlDataSourceErrorHandler.SqlTupleFailed("Can't find 'BATCH_CONSUMERS's!", sql, e)));
      }

      @Override
      public Uni<List<BatchConsumer>> findAllUsingSqlBuilder(FindAllFilter filter) {
        final var sql = registry.batchConsumers().findAllUsingSqlBuilder(filter);

        if(log.isDebugEnabled()) {
          log.debug("Batch2DbQueryImpl.queryBatchConsumers.findAllUsingSqlBuilder query, with props: {} \r\n{}",
            sql.getProps().deepToString(),
            sql.getValue());
        }

        @SuppressWarnings("unchecked")
        final TenantSql.RowMapper<BatchConsumer> mapper = (TenantSql.RowMapper<BatchConsumer>) sql.getRowMapper();

        return dataSource.getClient().preparedQuery(sql.getValue())
          .mapping(mapper::apply)
          .execute(sql.getProps())
          .onItem()
          .transformToUni((RowSet<BatchConsumer> rowset) -> Multi.createFrom().iterable(rowset).collect().asList())
          .onFailure().invoke(e -> errorHandler.deadEnd(new ThenaSqlDataSourceErrorHandler.SqlTupleFailed("Can't find 'BATCH_CONSUMERS's!", sql, e)));
      }

      @Override
      public Uni<List<BatchConsumer>> findAllByAppId(String appId, boolean lockForUpdate) {
        final var sql = registry.batchConsumers().findAllByAppId(appId, lockForUpdate);

        if(log.isDebugEnabled()) {
          log.debug("Batch2DbQueryImpl.queryBatchConsumers.findAllByAppId query, with props: {} \r\n{}",
            sql.getProps().deepToString(),
            sql.getValue());
        }

        @SuppressWarnings("unchecked")
        final TenantSql.RowMapper<BatchConsumer> mapper = (TenantSql.RowMapper<BatchConsumer>) sql.getRowMapper();

        return dataSource.getClient().preparedQuery(sql.getValue())
          .mapping(mapper::apply)
          .execute(sql.getProps())
          .onItem()
          .transformToUni((RowSet<BatchConsumer> rowset) -> Multi.createFrom().iterable(rowset).collect().asList())
          .onFailure().invoke(e -> errorHandler.deadEnd(new ThenaSqlDataSourceErrorHandler.SqlTupleFailed("Can't find 'BATCH_CONSUMERS's!", sql, e)));
      }
    };
  }
}
