package io.resys.thena.client.sample.spi;

import io.resys.thena.api.entities.Tenant;
import io.resys.thena.datasource.ThenaSqlDataSource;
import io.resys.thena.datasource.ThenaSqlDataSourceErrorHandler;
import io.resys.thena.registry.TenantRegistrySqlImpl;
import io.resys.thena.spi.InternalTenantQueryImpl;
import io.resys.thena.spi.TenantDataSource;
import io.resys.thena.support.RepoAssert;
import io.smallrye.mutiny.Uni;
import java.lang.Override;
import java.lang.StringBuilder;
import java.lang.System;
import lombok.extern.slf4j.Slf4j;

@Slf4j(
    topic = "io.resys.thena.show_sql"
)
public class Batch2DbInternalTenantQuery extends InternalTenantQueryImpl implements TenantDataSource.InternalTenantQuery {
  public Batch2DbInternalTenantQuery(ThenaSqlDataSource dataSource) {
    super(dataSource);
  }

  @Override
  public Uni<Tenant> insert(Tenant newRepo) {
    final var names = Batch2TableNames.defaults().toRepo(newRepo);
    final var next = dataSource.withTenant(newRepo);
    final var registry = new Batch2Registry(names, next);
    final var sqlQuery = new TenantRegistrySqlImpl(next.getTenantContext());
    final var pool = next.getPool();

    return pool.withTransaction(tx -> {
      final var tenantInsert = sqlQuery.insertOne(newRepo);
      final var tablesCreate = new StringBuilder();
      RepoAssert.isTrue(newRepo.getType() == Tenant.StructureType.batch, () -> "Tenant type must be batch");

      tablesCreate
        .append(registry.batchConsumers().createTable().getValue())

        .append(registry.batchConsumers().createConstraints().getValue())
        .toString();

      if(log.isDebugEnabled()) {
        log.debug(new StringBuilder("Creating schema: ")
          .append(System.lineSeparator())
          .append(tablesCreate.toString())
          .toString());
      }

      final Uni<Void> create = getClient().query(sqlQuery.createTable().getValue()).execute()
        .onItem().transformToUni(data -> Uni.createFrom().voidItem())
        .onFailure().invoke(e -> next.getErrorHandler().deadEnd(new ThenaSqlDataSourceErrorHandler.SqlFailed("Can't create table 'TENANT'!", sqlQuery.createTable(), e)));

      final Uni<Void> insert = tx.preparedQuery(tenantInsert.getValue()).execute(tenantInsert.getProps())
        .onItem().transformToUni(rowSet -> Uni.createFrom().voidItem())
        .onFailure().invoke(e -> next.getErrorHandler().deadEnd(new ThenaSqlDataSourceErrorHandler.SqlTupleFailed("Can't insert into 'TENANT'!", tenantInsert, e)));
      final Uni<Void> nested = tx.query(tablesCreate.toString()).execute()
        .onItem().transformToUni(rowSet -> Uni.createFrom().voidItem())
        .onFailure().invoke(e -> next.getErrorHandler().deadEnd(new ThenaSqlDataSourceErrorHandler.SqlSchemaFailed("Can't create tables!", tablesCreate.toString(), e)));

      return create
        .onItem().transformToUni((junk) -> insert)
        .onItem().transformToUni((junk) -> nested)
        .onItem().transform(junk -> newRepo)
        .onItem().invoke(newTenant -> this.dataSource.getTenantCache().setTenant(newTenant));
    });
  }

  @Override
  public Uni<Tenant> delete(Tenant newRepo) {
    final var names = Batch2TableNames.defaults().toRepo(newRepo);
    final var next = dataSource.withTenant(newRepo);
    final var registry = new Batch2Registry(names, next);
    final var sqlQuery = new TenantRegistrySqlImpl(next.getTenantContext());
    final var pool = next.getPool();

    return pool.withTransaction(tx -> {
      final var tenantDelete = sqlQuery.deleteOne(newRepo);
      final var tablesDrop = new StringBuilder();
      RepoAssert.isTrue(newRepo.getType() == Tenant.StructureType.batch, () -> "Tenant type must be batch");

      tablesDrop
        .append(registry.batchConsumers().dropTable().getValue())
        .toString();

      if(log.isDebugEnabled()) {
        log.debug("Delete tenant by name query, with props: {} \r\n{}",
          tenantDelete.getProps().deepToString(),
          tenantDelete.getValue());

        log.debug(new StringBuilder("Drop schema: ")
          .append(System.lineSeparator())
          .append(tablesDrop.toString())
          .toString());
      }

      final Uni<Void> insert = tx.preparedQuery(tenantDelete.getValue()).execute(tenantDelete.getProps())
        .onItem().transformToUni(rowSet -> Uni.createFrom().voidItem())
        .onFailure().invoke(e -> next.getErrorHandler().deadEnd(new ThenaSqlDataSourceErrorHandler.SqlTupleFailed("Can't delete from 'TENANT'!", tenantDelete, e)));
      final Uni<Void> nested = tx.query(tablesDrop.toString()).execute()
        .onItem().transformToUni(rowSet -> Uni.createFrom().voidItem())
        .onFailure().invoke(e -> next.getErrorHandler().deadEnd(new ThenaSqlDataSourceErrorHandler.SqlSchemaFailed("Can't drop tables!", tablesDrop.toString(), e)));

      return insert
        .onItem().transformToUni(junk -> nested)
        .onItem().transform(junk -> newRepo)
        .onItem().invoke(() -> this.dataSource.getTenantCache().invalidateAll());
    });
  }
}
