package io.resys.thena.client.sample.spi;

import io.resys.thena.api.actions.TenantActions;
import io.resys.thena.api.entities.Tenant;
import io.resys.thena.client.sample.Batch2Db;
import io.resys.thena.client.sample.Batch2DbBuilder;
import io.resys.thena.client.sample.Batch2DbQuery;
import io.resys.thena.datasource.TenantCacheImpl;
import io.resys.thena.datasource.TenantContext;
import io.resys.thena.datasource.ThenaDataSource;
import io.resys.thena.datasource.ThenaSqlDataSource;
import io.resys.thena.datasource.ThenaSqlDataSourceErrorHandler;
import io.resys.thena.datasource.ThenaSqlDataSourceImpl;
import io.resys.thena.datasource.vertx.ThenaSqlPoolVertx;
import io.resys.thena.spi.TenantActionsImpl;
import io.resys.thena.spi.TenantDataSource;
import io.resys.thena.spi.TenantException;
import io.resys.thena.support.RepoAssert;
import io.smallrye.mutiny.Uni;
import io.vertx.mutiny.sqlclient.Pool;
import java.lang.Override;
import java.lang.RuntimeException;
import java.lang.String;
import java.lang.StringBuilder;
import java.util.Optional;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

@RequiredArgsConstructor
@Slf4j
public class Batch2DbImpl implements Batch2Db {
  private final ThenaSqlDataSource dataSource;

  @Override
  public ThenaDataSource getDataSource() {
    return dataSource;
  }

  @Override
  public TenantDataSource.InternalTenantQuery tenant() {
    return new Batch2DbInternalTenantQuery(dataSource);
  }

  @Override
  public Uni<Batch2Db> withTenant(String tenantId) {
    return tenant().getByNameOrId(tenantId).onItem().transformToUni(tenant -> {
      if(tenant == null) {
        return tenantNotFound(tenantId);
      }
      return Uni.createFrom().item(withTenant(tenant));
    });
  }

  @Override
  public Batch2Db withTenant(Tenant tenant) {
    return new Batch2DbImpl(dataSource.withTenant(tenant));
  }

  @Override
  public Uni<Batch2Db> withTenant() {
    if(this.dataSource.isTenantLoaded()) {
      return Uni.createFrom().item(this);
    }
    return this.withTenant(this.dataSource.getTenant().getName());
  }

  @Override
  public <R> Uni<R> withTransaction(TenantDataSource.TxScope scope,
      Batch2Db.Transaction<R> callback) {
    return withTenant(scope.getTenantId()).onItem().transformToUni(state -> {
      final var source = (ThenaSqlDataSource) state.getDataSource();
      return source.getPool().withTransaction(conn -> callback.apply(new Batch2DbImpl(source.withTx(conn))));
    });
  }

  @Override
  public Batch2DbQuery query() {
    return new Batch2DbQueryImpl(dataSource);
  }

  @Override
  public Batch2DbBuilder builder() {
    return new Batch2DbBuilderImpl(dataSource);
  }

  public Uni<Batch2Db> createIfNot() {
    return tenant().findByNameOrId(this.dataSource.getTenant().getName())
      .onItem().transformToUni(repo -> {
        if(repo.isEmpty()) {
          return new TenantActionsImpl(this, Tenant.StructureType.batch)
            .commit()
            .name(this.dataSource.getTenant().getName())
            .build().onItem().transform(commit -> {
              if(commit.getStatus() != TenantActions.CommitStatus.OK) {
                final var msg = String.join(",", commit.getMessages().stream().map(e -> e.getText()).toList());
                final var ex = commit.getMessages().stream().map(e -> e.getException()).filter(e -> e != null).toList();
                throw new TenantException("Failed to create tenant: " + msg, ex);
              }
              return withTenant(commit.getRepo());
            });
        }
        return Uni.createFrom().item(withTenant(repo.get()));
      });
  }

  private <T> Uni<T> tenantNotFound(String tenantId) {
    return tenant().findAll().collect().asList().onItem().transform(tenants -> {
      final var text = new StringBuilder()
        .append("Tenant with name: '").append(tenantId).append("' does not exist!")
        .append(" known tenants: '").append(String.join(",", tenants.stream().map(r -> r.getName()).toList())).append("'")
        .toString();
      log.error(text);
      throw new RuntimeException(text);
    });
  }

  public static Batch2DbImpl create(TenantContext names, Pool client,
      ThenaSqlDataSource.TenantCache tenantCache, ThenaSqlDataSourceErrorHandler errorHandler) {
    final var pool = new ThenaSqlPoolVertx(client);
    final var dataSource = new ThenaSqlDataSourceImpl(
      "", names, pool, errorHandler,
      Optional.empty(),
      tenantCache
    );
    return new Batch2DbImpl(dataSource);
  }

  public static Builder create() {
    return new Builder();
  }

  public static class Builder {
    private Pool client;

    private String db;

    private ThenaSqlDataSourceErrorHandler errorHandler;

    private ThenaSqlDataSource.TenantCache tenantCache;

    public Builder errorHandler(ThenaSqlDataSourceErrorHandler errorHandler) {
      this.errorHandler = errorHandler;
      return this;
    }

    public Builder tenant(String db) {
      this.db = db;
      return this;
    }

    public Builder tenantCache(ThenaSqlDataSource.TenantCache tenantCache) {
      this.tenantCache = tenantCache;
      return this;
    }

    public Builder client(Pool client) {
      this.client = client;
      return this;
    }

    public Batch2DbImpl build() {
      RepoAssert.notNull(client, () -> "client must be defined!");
      RepoAssert.notNull(db, () -> "db must be defined!");
      RepoAssert.notNull(errorHandler, () -> "errorHandler must be defined!");

      final var tenantCache = this.tenantCache == null ? new TenantCacheImpl() : this.tenantCache;
      final var ctx = TenantContext.defaults(db);
      final var pool = new ThenaSqlPoolVertx(client);

      final var dataSource = new ThenaSqlDataSourceImpl(
        db, ctx, pool, errorHandler,
        Optional.empty(),
        tenantCache
      );
      return new Batch2DbImpl(dataSource);
    }
  }
}
