package io.resys.thena.client.sample.spi;

import io.resys.thena.api.entities.BatchStatus;
import io.resys.thena.api.entities.ImmutableBatchLog;
import io.resys.thena.client.sample.Batch2DbBuilder;
import io.resys.thena.client.sample.ImmutablePersistenceUnit;
import io.resys.thena.client.sample.entities.BatchConsumer;
import io.resys.thena.datasource.ThenaSqlClient;
import io.resys.thena.datasource.ThenaSqlDataSource;
import io.resys.thena.storesql.support.Execute;
import io.smallrye.mutiny.Uni;
import java.lang.Class;
import java.lang.Override;
import java.lang.RuntimeException;
import java.lang.String;
import java.lang.StringBuilder;
import java.lang.System;
import java.lang.Throwable;
import java.util.ArrayList;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class Batch2DbBuilderImpl implements Batch2DbBuilder {
  private static final Logger log = LoggerFactory.getLogger("io.resys.thena.client.sample.batch2.show_sql");

  private final ThenaSqlClient tx;

  private final ThenaSqlDataSource dataSource;

  private final Batch2Registry registry;

  private final StringBuilder txLog;

  private final ImmutablePersistenceUnit.Builder init = ImmutablePersistenceUnit.builder();

  public Batch2DbBuilderImpl(ThenaSqlDataSource dataSource) {
    final var names = Batch2TableNames.defaults().toRepo(dataSource.getTenant());
    this.registry = new Batch2Registry(names, dataSource);
    this.dataSource = dataSource;
    this.tx = dataSource.getClient();
    this.txLog = new StringBuilder();
  }

  @Override
  public Batch2DbBuilder from(Batch2DbBuilder.PersistenceUnit unit) {
    init.from(unit);
    return this;
  }

  @Override
  public Uni<Batch2DbBuilder.PersistenceUnit> persist() {
    final var entries = init.build();

    return Uni.combine().all()
      .unis(
        // === DELETE OPERATIONS ===
        visitBatchConsumersDeletes(entries), // 1

        // === INSERT OPERATIONS ===
        visitBatchConsumersInserts(entries), // 2

        // === UPDATE OPERATIONS ===
        visitBatchConsumersUpdates(entries) // 3
      )
      .with(Batch2DbBuilder.PersistenceUnit.class, (items) -> visitSuccess(entries, items))
      .onFailure(Batch2BuilderException.class)
      .recoverWithUni(this::visitError);
  }

  private Uni<Batch2DbBuilder.PersistenceUnit> visitBatchConsumersDeletes(
      Batch2DbBuilder.PersistenceUnit entries) {
    final var data = entries.getBatchConsumersDeletes();
    final var sql = registry.batchConsumers().deleteAll(data);
    return visitExecution(sql, BatchConsumer.class);
  }

  private Uni<Batch2DbBuilder.PersistenceUnit> visitBatchConsumersInserts(
      Batch2DbBuilder.PersistenceUnit entries) {
    final var data = entries.getBatchConsumersInserts();
    final var sql = registry.batchConsumers().insertMany(data);
    return visitExecution(sql, BatchConsumer.class);
  }

  private Uni<Batch2DbBuilder.PersistenceUnit> visitBatchConsumersUpdates(
      Batch2DbBuilder.PersistenceUnit entries) {
    final var data = entries.getBatchConsumersUpdates();
    final var sql = registry.batchConsumers().updateMany(data);
    return visitExecution(sql, BatchConsumer.class);
  }

  private Uni<Batch2DbBuilder.PersistenceUnit> visitExecution(ThenaSqlClient.SqlTupleList sql,
      Class<?> type) {
    visitTxLog(sql, type);

    final var container = ImmutablePersistenceUnit.builder()
          .tenantId(this.dataSource.getTenant().getId())
          .status(BatchStatus.OK)
          .log("")
          .build();

    return Execute.apply(tx, sql)
      .onItem().transform(row -> {
        final var text = "Inserted " + (row == null ? 0 : row.rowCount()) + " " + type.getSimpleName() + " entries";
        final var updatedMessages = new ArrayList<>(container.getCommitLogs());
        updatedMessages.add(ImmutableBatchLog.builder().text(text).build());
        return (Batch2DbBuilder.PersistenceUnit) ImmutablePersistenceUnit.builder().from(container)
          .commitLogs(updatedMessages)
          .build();
      })
      .onFailure().transform(t -> {
        final var text = "Failed to insert " + sql.getProps().size() + " " + type.getSimpleName() + " entries";
        return new Batch2BuilderException(container, text, t);
      });
  }

  private void visitTxLog(ThenaSqlClient.SqlTupleList sql, Class<?> type) {
    if(sql.getProps().isEmpty()) {
      return;
    }

    this.txLog
          .append(System.lineSeparator())
          .append("--- processing ").append(sql.getProps().size()).append(" entries of type: '").append(type.getSimpleName()).append("'")
          .append(sql.getPropsDeepString()).append(System.lineSeparator())
          .append(sql.getValue()).append(System.lineSeparator());
  }

  private Batch2DbBuilder.PersistenceUnit visitSuccess(
      Batch2DbBuilder.PersistenceUnit inputContainer, List<Batch2DbBuilder.PersistenceUnit> items) {
    final var msg = System.lineSeparator() + "--- TX LOG" + System.lineSeparator() + txLog;
    if(log.isDebugEnabled()) {
      log.debug(msg);
    }

    return inputContainer.merge(items.stream()
          .reduce((a, b) -> a.merge(b))
          .orElse(inputContainer));
  }

  private Uni<Batch2DbBuilder.PersistenceUnit> visitError(Throwable ex) {
    final var msg = System.lineSeparator() + "--- TX LOG" + System.lineSeparator() + txLog;
    final var builderError = (Batch2BuilderException) ex;
    log.error("Failed to persist because of: {},\r\n{}", ex.getMessage(), msg, ex);

    return tx.rollback().onItem().transform(junk -> 
          ImmutablePersistenceUnit.builder().from(builderError.getContainer())
            .log(msg)
            .build()
        );
  }

  public static final class Batch2BuilderException extends RuntimeException {
    private final Batch2DbBuilder.PersistenceUnit container;

    public Batch2BuilderException(Batch2DbBuilder.PersistenceUnit container, String message,
        Throwable cause) {
      super(message, cause);
      this.container = container;
    }

    public Batch2DbBuilder.PersistenceUnit getContainer() {
      return container;
    }
  }
}
