/*
 * Decompiled with CFR 0.152.
 */
package io.deephaven.server.session;

import com.github.f4b6a3.uuid.UuidCreator;
import com.github.f4b6a3.uuid.exception.InvalidUuidException;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.protobuf.ByteString;
import com.google.rpc.Code;
import io.deephaven.auth.AuthContext;
import io.deephaven.auth.AuthenticationException;
import io.deephaven.auth.AuthenticationRequestHandler;
import io.deephaven.configuration.Configuration;
import io.deephaven.extensions.barrage.util.GrpcUtil;
import io.deephaven.internal.log.LoggerFactory;
import io.deephaven.io.logger.Logger;
import io.deephaven.proto.backplane.grpc.TerminationNotificationResponse;
import io.deephaven.proto.util.Exceptions;
import io.deephaven.server.session.DelegatingSessionListener;
import io.deephaven.server.session.SessionCloseableObserver;
import io.deephaven.server.session.SessionListener;
import io.deephaven.server.session.SessionServiceGrpcImpl;
import io.deephaven.server.session.SessionState;
import io.deephaven.server.util.Scheduler;
import io.deephaven.util.process.ProcessEnvironment;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import io.grpc.stub.StreamObserver;
import java.util.Arrays;
import java.util.Deque;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.inject.Named;
import javax.inject.Singleton;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.VisibleForTesting;

@Singleton
public class SessionService {
    private static final Logger log = LoggerFactory.getLogger(SessionService.class);
    static final long MIN_COOKIE_EXPIRE_MS = 10000L;
    private static final int MAX_STACK_TRACE_CAUSAL_DEPTH = Configuration.getInstance().getIntegerForClassWithDefault(SessionService.class, "maxStackTraceCausedByDepth", 20);
    private static final int MAX_STACK_TRACE_DEPTH = Configuration.getInstance().getIntegerForClassWithDefault(SessionService.class, "maxStackTraceDepth", 50);
    private final Scheduler scheduler;
    private final SessionState.Factory sessionFactory;
    private final long tokenExpireMs;
    private final long tokenRotateMs;
    private final Map<UUID, TokenExpiration> tokenToSession = new ConcurrentHashMap<UUID, TokenExpiration>();
    private final Deque<TokenExpiration> outstandingCookies = new ConcurrentLinkedDeque<TokenExpiration>();
    private boolean cleanupJobInstalled = false;
    private final SessionCleanupJob sessionCleanupJob = new SessionCleanupJob();
    private final List<TerminationNotificationListener> terminationListeners = new CopyOnWriteArrayList<TerminationNotificationListener>();
    private final Map<String, AuthenticationRequestHandler> authRequestHandlers;
    private final SessionListener sessionListener;

    @Inject
    public SessionService(Scheduler scheduler, SessionState.Factory sessionFactory, @Named(value="session.tokenExpireMs") long tokenExpireMs, Map<String, AuthenticationRequestHandler> authRequestHandlers, Set<SessionListener> sessionListeners) {
        this.scheduler = scheduler;
        this.sessionFactory = sessionFactory;
        this.tokenExpireMs = tokenExpireMs;
        this.authRequestHandlers = authRequestHandlers;
        if (tokenExpireMs < 10000L) {
            throw new IllegalArgumentException("session.tokenExpireMs is set too low. It is configured to " + tokenExpireMs + "ms (minimum is 10000ms). At low levels it is difficult to guarantee smooth operability given a distributed system and potential clock drift");
        }
        long tokenExpireNanos = TimeUnit.MILLISECONDS.toNanos(tokenExpireMs);
        if (tokenExpireNanos == Long.MIN_VALUE || tokenExpireNanos == Long.MAX_VALUE) {
            throw new IllegalArgumentException("session.tokenExpireMs is set too high.");
        }
        this.tokenRotateMs = tokenExpireMs / 5L;
        if (ProcessEnvironment.tryGet() != null) {
            ProcessEnvironment.getGlobalFatalErrorReporter().addInterceptor(this::onFatalError);
        }
        this.sessionListener = new DelegatingSessionListener(sessionListeners);
    }

    private synchronized void onFatalError(@NotNull String message, @NotNull Throwable throwable, boolean isFromUncaught) {
        TerminationNotificationResponse.Builder builder = TerminationNotificationResponse.newBuilder().setAbnormalTermination(true).setIsFromUncaughtException(isFromUncaught).setReason(message);
        for (int depth = 0; throwable != null && depth < MAX_STACK_TRACE_CAUSAL_DEPTH; throwable = throwable.getCause(), ++depth) {
            builder.addStackTraces(SessionService.transformToProtoBuf(throwable));
        }
        TerminationNotificationResponse notification = builder.build();
        this.terminationListeners.forEach(listener -> listener.sendMessage(notification));
        this.terminationListeners.clear();
    }

    private static TerminationNotificationResponse.StackTrace transformToProtoBuf(@NotNull Throwable throwable) {
        return TerminationNotificationResponse.StackTrace.newBuilder().setType(throwable.getClass().getName()).setMessage(Objects.toString(throwable.getMessage())).addAllElements((Iterable)Arrays.stream(throwable.getStackTrace()).limit(MAX_STACK_TRACE_DEPTH).map(StackTraceElement::toString).collect(Collectors.toList())).build();
    }

    public synchronized void onShutdown() {
        TerminationNotificationResponse notification = TerminationNotificationResponse.newBuilder().setAbnormalTermination(false).build();
        this.terminationListeners.forEach(listener -> listener.sendMessage(notification));
        this.terminationListeners.clear();
        this.closeAllSessions();
    }

    public void addTerminationListener(SessionState session, StreamObserver<TerminationNotificationResponse> responseObserver) {
        this.terminationListeners.add(new TerminationNotificationListener(session, responseObserver));
    }

    public SessionState newSession(AuthContext authContext) {
        SessionState session = this.sessionFactory.create(authContext);
        this.checkTokenAndRotate(session, true);
        this.sessionListener.onSessionCreate(session);
        return session;
    }

    public TokenExpiration refreshToken(SessionState session) {
        return this.checkTokenAndRotate(session, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private TokenExpiration checkTokenAndRotate(SessionState session, boolean initialToken) {
        TokenExpiration expiration;
        long nowMillis = this.scheduler.currentTimeMillis();
        Object object = session;
        synchronized (object) {
            UUID newUUID;
            expiration = session.getExpiration();
            if (!initialToken) {
                if (expiration == null) {
                    return null;
                }
                if (expiration.deadlineMillis - this.tokenExpireMs + this.tokenRotateMs > nowMillis) {
                    return expiration;
                }
            }
            while (this.tokenToSession.putIfAbsent(newUUID = UuidCreator.getRandomBased(), expiration = new TokenExpiration(newUUID, nowMillis + this.tokenExpireMs, session)) != null) {
            }
            if (initialToken) {
                session.initializeExpiration(expiration);
            } else {
                session.updateExpiration(expiration);
            }
        }
        this.outstandingCookies.addLast(expiration);
        object = this;
        synchronized (object) {
            if (!this.cleanupJobInstalled) {
                this.cleanupJobInstalled = true;
                this.scheduler.runAtTime(expiration.deadlineMillis, this.sessionCleanupJob);
            }
        }
        return expiration;
    }

    public long getExpirationDelayMs() {
        return this.tokenExpireMs;
    }

    public SessionState getSessionForAuthToken(String token) throws AuthenticationException {
        Optional s;
        int offset;
        String bearerKey = null;
        String bearerPayload = null;
        if (token.startsWith("Bearer ")) {
            String authToken = token.substring("Bearer ".length());
            try {
                UUID uuid = UuidCreator.fromString((String)authToken);
                SessionState session = this.getSessionForToken(uuid);
                if (session != null) {
                    return session;
                }
            }
            catch (InvalidUuidException uuid) {
                // empty catch block
            }
            int offset2 = authToken.indexOf(32);
            bearerKey = authToken.substring(0, offset2 < 0 ? authToken.length() : offset2);
            bearerPayload = offset2 < 0 ? "" : authToken.substring(offset2 + 1);
        }
        String key = token.substring(0, (offset = token.indexOf(32)) < 0 ? token.length() : offset);
        String payload = offset < 0 ? "" : token.substring(offset + 1);
        AuthenticationRequestHandler handler = this.authRequestHandlers.get(key);
        if (handler != null && (s = handler.login(payload, SessionServiceGrpcImpl::insertCallHeader)).isPresent()) {
            return this.newSession((AuthContext)s.get());
        }
        if (bearerKey != null && (handler = this.authRequestHandlers.get(bearerKey)) != null && (s = handler.login(bearerPayload, SessionServiceGrpcImpl::insertCallHeader)).isPresent()) {
            return this.newSession((AuthContext)s.get());
        }
        log.info().append((CharSequence)"No AuthenticationRequestHandler registered for type ").append((CharSequence)key).endl();
        throw new AuthenticationException();
    }

    public SessionState getSessionForToken(UUID token) {
        TokenExpiration expiration = this.tokenToSession.get(token);
        if (expiration == null || expiration.session.isExpired() || expiration.deadlineMillis <= this.scheduler.currentTimeMillis()) {
            return null;
        }
        return expiration.session;
    }

    @NotNull
    public SessionState getCurrentSession() {
        SessionState session = this.getOptionalSession();
        if (session == null) {
            throw Status.UNAUTHENTICATED.asRuntimeException();
        }
        return session;
    }

    @Nullable
    public SessionState getOptionalSession() {
        SessionState session = (SessionState)SessionServiceGrpcImpl.SESSION_CONTEXT_KEY.get();
        if (session == null || session.isExpired()) {
            return null;
        }
        return session;
    }

    public void closeSession(SessionState session) {
        if (session.isExpired()) {
            return;
        }
        session.onExpired();
    }

    public void closeAllSessions() {
        for (TokenExpiration token : this.outstandingCookies) {
            token.session.onExpired();
        }
    }

    private final class SessionCleanupJob
    implements Runnable {
        private SessionCleanupJob() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            TokenExpiration next;
            long nowMillis = SessionService.this.scheduler.currentTimeMillis();
            while ((next = SessionService.this.outstandingCookies.peek()) != null && next.deadlineMillis <= nowMillis) {
                SessionService.this.outstandingCookies.poll();
                if (!next.session.isExpired()) continue;
                next.session.onExpired();
            }
            SessionService sessionService = SessionService.this;
            synchronized (sessionService) {
                TokenExpiration next2 = SessionService.this.outstandingCookies.peek();
                if (next2 == null) {
                    SessionService.this.cleanupJobInstalled = false;
                } else {
                    SessionService.this.scheduler.runAtTime(next2.deadlineMillis, this);
                }
            }
        }
    }

    private final class TerminationNotificationListener
    extends SessionCloseableObserver<TerminationNotificationResponse> {
        public TerminationNotificationListener(SessionState session, StreamObserver<TerminationNotificationResponse> responseObserver) {
            super(session, responseObserver);
        }

        @Override
        protected void onClose() {
            GrpcUtil.safelyError((StreamObserver)this.responseObserver, (Code)Code.UNAUTHENTICATED, (String)"Session has ended");
            SessionService.this.terminationListeners.remove(this);
        }

        void sendMessage(TerminationNotificationResponse response) {
            GrpcUtil.safelyOnNextAndComplete((StreamObserver)this.responseObserver, (Object)response);
        }
    }

    public static final class TokenExpiration {
        public final UUID token;
        public final long deadlineMillis;
        public final SessionState session;

        public TokenExpiration(UUID cookie, long deadlineMillis, SessionState session) {
            this.token = cookie;
            this.deadlineMillis = deadlineMillis;
            this.session = session;
        }

        public ByteString getBearerTokenAsByteString() {
            return ByteString.copyFromUtf8((String)("Bearer " + UuidCreator.toString((UUID)this.token)));
        }

        public ByteString getTokenAsByteString() {
            return ByteString.copyFromUtf8((String)UuidCreator.toString((UUID)this.token));
        }
    }

    @Singleton
    public static class ObfuscatingErrorTransformer
    implements ErrorTransformer {
        @VisibleForTesting
        static final int MAX_STACK_TRACE_CAUSAL_DEPTH = 25;
        private static final int MAX_CACHE_BUILDER_SIZE = 1009;
        private static final int MAX_CACHE_DURATION_MIN = 1;
        private final Cache<Throwable, UUID> idCache = CacheBuilder.newBuilder().expireAfterAccess(1L, TimeUnit.MINUTES).maximumSize(1009L).weakKeys().build();

        @Inject
        public ObfuscatingErrorTransformer() {
        }

        @Override
        public StatusRuntimeException transform(Throwable err) {
            if (err instanceof StatusRuntimeException) {
                StatusRuntimeException sre = (StatusRuntimeException)err;
                if (sre.getStatus().getCode().equals((Object)Status.UNAUTHENTICATED.getCode())) {
                    log.debug().append((CharSequence)"ignoring unauthenticated request").endl();
                } else if (sre.getStatus().getCode().equals((Object)Status.CANCELLED.getCode())) {
                    log.debug().append((CharSequence)"ignoring cancelled request").endl();
                } else {
                    log.debug().append((Throwable)sre).endl();
                }
                return sre;
            }
            if (err instanceof InterruptedException) {
                return this.securelyWrapError(err, Code.UNAVAILABLE);
            }
            return this.securelyWrapError(err, Code.INVALID_ARGUMENT);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private StatusRuntimeException securelyWrapError(@NotNull Throwable err, Code statusCode) {
            boolean shouldLog;
            UUID errorId;
            Cache<Throwable, UUID> cache = this.idCache;
            synchronized (cache) {
                errorId = (UUID)this.idCache.getIfPresent((Object)err);
                shouldLog = errorId == null;
                int currDepth = 0;
                for (Throwable causeToCheck = err.getCause(); errorId == null && ++currDepth < 25 && causeToCheck != null; causeToCheck = causeToCheck.getCause()) {
                    errorId = (UUID)this.idCache.getIfPresent((Object)causeToCheck);
                }
                if (errorId == null) {
                    errorId = UuidCreator.getRandomBased();
                }
                Throwable throwableToAdd = err;
                while (currDepth > 0) {
                    if (throwableToAdd.getStackTrace().length > 0) {
                        this.idCache.put((Object)throwableToAdd, (Object)errorId);
                    }
                    throwableToAdd = throwableToAdd.getCause();
                    --currDepth;
                }
            }
            if (shouldLog) {
                log.error().append((CharSequence)"Internal Error '").append((CharSequence)errorId.toString()).append((CharSequence)"' ").append(err).endl();
            }
            return Exceptions.statusRuntimeException((Code)statusCode, (String)("Details Logged w/ID '" + String.valueOf(errorId) + "'"));
        }
    }

    @FunctionalInterface
    public static interface ErrorTransformer {
        public StatusRuntimeException transform(Throwable var1);
    }
}

