/*
 * Decompiled with CFR 0.152.
 */
package io.deephaven.server.notebook;

import com.google.common.hash.HashFunction;
import com.google.common.hash.Hashing;
import com.google.common.io.ByteSource;
import com.google.protobuf.ByteString;
import com.google.rpc.Code;
import io.deephaven.configuration.Configuration;
import io.deephaven.configuration.DataDir;
import io.deephaven.internal.log.LoggerFactory;
import io.deephaven.io.logger.Logger;
import io.deephaven.proto.backplane.grpc.CreateDirectoryRequest;
import io.deephaven.proto.backplane.grpc.CreateDirectoryResponse;
import io.deephaven.proto.backplane.grpc.DeleteItemRequest;
import io.deephaven.proto.backplane.grpc.DeleteItemResponse;
import io.deephaven.proto.backplane.grpc.FetchFileRequest;
import io.deephaven.proto.backplane.grpc.FetchFileResponse;
import io.deephaven.proto.backplane.grpc.ItemInfo;
import io.deephaven.proto.backplane.grpc.ItemType;
import io.deephaven.proto.backplane.grpc.ListItemsRequest;
import io.deephaven.proto.backplane.grpc.ListItemsResponse;
import io.deephaven.proto.backplane.grpc.MoveItemRequest;
import io.deephaven.proto.backplane.grpc.MoveItemResponse;
import io.deephaven.proto.backplane.grpc.SaveFileRequest;
import io.deephaven.proto.backplane.grpc.SaveFileResponse;
import io.deephaven.proto.backplane.grpc.StorageServiceGrpc;
import io.deephaven.proto.util.Exceptions;
import io.deephaven.server.session.SessionService;
import io.grpc.stub.StreamObserver;
import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.regex.PatternSyntaxException;
import java.util.stream.Stream;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.jetbrains.annotations.NotNull;

@Singleton
public class FilesystemStorageServiceGrpcImpl
extends StorageServiceGrpc.StorageServiceImplBase {
    private static final Logger log = LoggerFactory.getLogger(FilesystemStorageServiceGrpcImpl.class);
    private static final String STORAGE_PATH = Configuration.getInstance().getStringWithDefault("storage.path", DataDir.get().resolve("storage").toString());
    private static final String WEB_LAYOUT_DIRECTORY = Configuration.getInstance().getProperty("web.storage.layout.directory");
    private static final String WEB_NOTEBOOK_DIRECTORY = Configuration.getInstance().getProperty("web.storage.notebook.directory");
    private static final String[] PRE_CREATE_PATHS = Configuration.getInstance().getStringArrayFromPropertyWithDefault("storage.path.defaults", new String[]{WEB_LAYOUT_DIRECTORY, WEB_NOTEBOOK_DIRECTORY});
    private static final HashFunction HASH_FUNCTION = Hashing.murmur3_128((int)0);
    @Deprecated
    private static final String REQUIRED_PATH_PREFIX = "/";
    private final Path root = Paths.get(STORAGE_PATH, new String[0]).normalize();
    private final SessionService sessionService;
    private final SessionService.ErrorTransformer errorTransformer;

    @Inject
    public FilesystemStorageServiceGrpcImpl(SessionService sessionService, SessionService.ErrorTransformer errorTransformer) {
        this.sessionService = sessionService;
        this.errorTransformer = errorTransformer;
        try {
            Files.createDirectories(this.root, new FileAttribute[0]);
            for (String path : PRE_CREATE_PATHS) {
                Files.createDirectories(this.resolveOrThrow(path), new FileAttribute[0]);
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException("Failed to initialize storage", e);
        }
    }

    private Path resolveOrThrow(String incomingPath) {
        Path resolved;
        if (incomingPath.startsWith(File.separator)) {
            incomingPath = incomingPath.substring(1);
        }
        if ((resolved = this.root.resolve(incomingPath).normalize()).startsWith(this.root)) {
            return resolved;
        }
        throw Exceptions.statusRuntimeException((Code)Code.INVALID_ARGUMENT, (String)("Invalid path: " + incomingPath));
    }

    private void requireNotRoot(Path path, String message) {
        if (path.equals(this.root)) {
            throw Exceptions.statusRuntimeException((Code)Code.FAILED_PRECONDITION, (String)message);
        }
    }

    public void listItems(@NotNull ListItemsRequest request, @NotNull StreamObserver<ListItemsResponse> responseObserver) {
        this.sessionService.getCurrentSession();
        ListItemsResponse.Builder builder = ListItemsResponse.newBuilder();
        PathMatcher matcher = request.hasFilterGlob() ? FilesystemStorageServiceGrpcImpl.createPathFilter(request.getFilterGlob()) : ignore -> true;
        Path dir = this.resolveOrThrow(request.getPath());
        builder.setCanonicalPath(REQUIRED_PATH_PREFIX + String.valueOf(this.root.relativize(dir)));
        try (Stream<Path> list = Files.list(dir);){
            for (Path p : list::iterator) {
                if (!matcher.matches(dir.relativize(p))) continue;
                BasicFileAttributes attrs = Files.readAttributes(p, BasicFileAttributes.class, new LinkOption[0]);
                boolean isDirectory = attrs.isDirectory();
                ItemInfo.Builder info = ItemInfo.newBuilder().setPath(REQUIRED_PATH_PREFIX + String.valueOf(this.root.relativize(p)));
                if (isDirectory) {
                    info.setType(ItemType.DIRECTORY);
                } else {
                    info.setSize(attrs.size()).setEtag(FilesystemStorageServiceGrpcImpl.hash(p)).setType(ItemType.FILE);
                }
                builder.addItems(info.build());
            }
        }
        catch (NoSuchFileException noSuchFileException) {
            throw Exceptions.statusRuntimeException((Code)Code.FAILED_PRECONDITION, (String)"Directory does not exist");
        }
        catch (IOException ioe) {
            throw this.errorTransformer.transform(ioe);
        }
        responseObserver.onNext((Object)builder.build());
        responseObserver.onCompleted();
    }

    private static PathMatcher createPathFilter(String filterGlob) {
        if (filterGlob.contains("**")) {
            throw Exceptions.statusRuntimeException((Code)Code.INVALID_ARGUMENT, (String)"Bad glob, only single `*`s are supported");
        }
        if (filterGlob.contains(File.separator)) {
            throw Exceptions.statusRuntimeException((Code)Code.INVALID_ARGUMENT, (String)"Bad glob, only the same directory can be checked");
        }
        try {
            return FileSystems.getDefault().getPathMatcher("glob:" + filterGlob);
        }
        catch (PatternSyntaxException e) {
            throw Exceptions.statusRuntimeException((Code)Code.INVALID_ARGUMENT, (String)("Bad glob, can't parse expression: " + e.getMessage()));
        }
    }

    private static String hash(Path path) throws IOException {
        return com.google.common.io.Files.asByteSource((File)path.toFile()).hash(HASH_FUNCTION).toString();
    }

    public void fetchFile(@NotNull FetchFileRequest request, @NotNull StreamObserver<FetchFileResponse> responseObserver) {
        String etag;
        byte[] bytes;
        this.sessionService.getCurrentSession();
        try {
            bytes = Files.readAllBytes(this.resolveOrThrow(request.getPath()));
            etag = ByteSource.wrap((byte[])bytes).hash(HASH_FUNCTION).toString();
        }
        catch (NoSuchFileException noSuchFileException) {
            throw Exceptions.statusRuntimeException((Code)Code.FAILED_PRECONDITION, (String)"File does not exist");
        }
        catch (IOException ioe) {
            throw this.errorTransformer.transform(ioe);
        }
        FetchFileResponse.Builder response = FetchFileResponse.newBuilder();
        response.setEtag(etag);
        if (!request.hasEtag() || !etag.equals(request.getEtag())) {
            response.setContents(ByteString.copyFrom((byte[])bytes));
        }
        responseObserver.onNext((Object)response.build());
        responseObserver.onCompleted();
    }

    public void saveFile(@NotNull SaveFileRequest request, @NotNull StreamObserver<SaveFileResponse> responseObserver) {
        String etag;
        this.sessionService.getCurrentSession();
        Path path = this.resolveOrThrow(request.getPath());
        this.requireNotRoot(path, "Can't overwrite the root directory");
        StandardOpenOption option = request.getAllowOverwrite() ? StandardOpenOption.TRUNCATE_EXISTING : StandardOpenOption.CREATE_NEW;
        byte[] bytes = request.getContents().toByteArray();
        try {
            etag = ByteSource.wrap((byte[])bytes).hash(HASH_FUNCTION).toString();
            Files.write(path, bytes, StandardOpenOption.CREATE, option);
        }
        catch (FileAlreadyExistsException alreadyExistsException) {
            throw Exceptions.statusRuntimeException((Code)Code.FAILED_PRECONDITION, (String)"File already exists");
        }
        catch (NoSuchFileException noSuchFileException) {
            throw Exceptions.statusRuntimeException((Code)Code.FAILED_PRECONDITION, (String)"Directory does not exist");
        }
        catch (IOException ioe) {
            throw this.errorTransformer.transform(ioe);
        }
        responseObserver.onNext((Object)SaveFileResponse.newBuilder().setEtag(etag).build());
        responseObserver.onCompleted();
    }

    public void moveItem(@NotNull MoveItemRequest request, @NotNull StreamObserver<MoveItemResponse> responseObserver) {
        CopyOption[] copyOptionArray;
        this.sessionService.getCurrentSession();
        Path source = this.resolveOrThrow(request.getOldPath());
        Path target = this.resolveOrThrow(request.getNewPath());
        this.requireNotRoot(target, "Can't overwrite the root directory");
        if (request.getAllowOverwrite()) {
            StandardCopyOption[] standardCopyOptionArray = new StandardCopyOption[1];
            copyOptionArray = standardCopyOptionArray;
            standardCopyOptionArray[0] = StandardCopyOption.REPLACE_EXISTING;
        } else {
            copyOptionArray = new StandardCopyOption[]{};
        }
        CopyOption[] options = copyOptionArray;
        try {
            Files.move(source, target, options);
        }
        catch (NoSuchFileException noSuchFileException) {
            throw Exceptions.statusRuntimeException((Code)Code.FAILED_PRECONDITION, (String)"File does not exist, cannot rename");
        }
        catch (FileAlreadyExistsException alreadyExistsException) {
            throw Exceptions.statusRuntimeException((Code)Code.FAILED_PRECONDITION, (String)"File already exists, cannot rename to replace");
        }
        catch (DirectoryNotEmptyException directoryNotEmptyException) {
            throw Exceptions.statusRuntimeException((Code)Code.FAILED_PRECONDITION, (String)"Cannot replace non-empty directory");
        }
        catch (IOException ioe) {
            throw this.errorTransformer.transform(ioe);
        }
        responseObserver.onNext((Object)MoveItemResponse.getDefaultInstance());
        responseObserver.onCompleted();
    }

    public void createDirectory(@NotNull CreateDirectoryRequest request, @NotNull StreamObserver<CreateDirectoryResponse> responseObserver) {
        this.sessionService.getCurrentSession();
        Path dir = this.resolveOrThrow(request.getPath());
        this.requireNotRoot(dir, "Can't overwrite the root directory");
        try {
            Files.createDirectory(dir, new FileAttribute[0]);
        }
        catch (FileAlreadyExistsException fileAlreadyExistsException) {
            throw Exceptions.statusRuntimeException((Code)Code.FAILED_PRECONDITION, (String)"Something already exists with that name");
        }
        catch (NoSuchFileException noSuchFileException) {
            throw Exceptions.statusRuntimeException((Code)Code.FAILED_PRECONDITION, (String)"Can't create directory, parent directory doesn't exist");
        }
        catch (IOException ioe) {
            throw this.errorTransformer.transform(ioe);
        }
        responseObserver.onNext((Object)CreateDirectoryResponse.getDefaultInstance());
        responseObserver.onCompleted();
    }

    public void deleteItem(@NotNull DeleteItemRequest request, @NotNull StreamObserver<DeleteItemResponse> responseObserver) {
        this.sessionService.getCurrentSession();
        Path path = this.resolveOrThrow(request.getPath());
        this.requireNotRoot(path, "Can't delete the root directory");
        try {
            Files.delete(path);
        }
        catch (NoSuchFileException noSuchFileException) {
            throw Exceptions.statusRuntimeException((Code)Code.FAILED_PRECONDITION, (String)"Cannot delete, file does not exists");
        }
        catch (DirectoryNotEmptyException directoryNotEmptyException) {
            throw Exceptions.statusRuntimeException((Code)Code.FAILED_PRECONDITION, (String)"Cannot delete non-empty directory");
        }
        catch (IOException ioe) {
            throw this.errorTransformer.transform(ioe);
        }
        responseObserver.onNext((Object)DeleteItemResponse.getDefaultInstance());
        responseObserver.onCompleted();
    }
}

