/*
 * Decompiled with CFR 0.152.
 */
package io.deephaven.time.calendar;

import io.deephaven.base.verify.Require;
import io.deephaven.time.DateTimeUtils;
import io.deephaven.time.calendar.Calendar;
import io.deephaven.time.calendar.CalendarDay;
import io.deephaven.time.calendar.ReadOptimizedConcurrentCache;
import io.deephaven.time.calendar.YearMonthSummaryCache;
import java.time.DayOfWeek;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

public class BusinessCalendar
extends Calendar {
    private final LocalDate firstValidDate;
    private final LocalDate lastValidDate;
    private final CalendarDay<LocalTime> standardBusinessDay;
    private final Set<DayOfWeek> weekendDays;
    private final Map<LocalDate, CalendarDay<Instant>> holidays;
    private final ReadOptimizedConcurrentCache<ReadOptimizedConcurrentCache.Pair<CalendarDay<Instant>>> schedulesCache = new ReadOptimizedConcurrentCache<ReadOptimizedConcurrentCache.Pair>(10000, this::computeCalendarDay);
    private final YearMonthSummaryCache<SummaryData> summaryCache = new YearMonthSummaryCache<SummaryData>(this::computeMonthSummary, this::computeYearSummary);
    private final int yearCacheStart;
    private final int yearCacheEnd;

    @Override
    synchronized void clearCache() {
        super.clearCache();
        this.schedulesCache.clear();
        this.summaryCache.clear();
    }

    private SummaryData summarize(int key, LocalDate startDate, LocalDate endDate) {
        ZonedDateTime start = startDate.atTime(0, 0).atZone(this.timeZone());
        ZonedDateTime end = endDate.atTime(0, 0).atZone(this.timeZone());
        LocalDate date = startDate;
        long businessTimeNanos = 0L;
        int businessDays = 0;
        int nonBusinessDays = 0;
        ArrayList<LocalDate> businessDates = new ArrayList<LocalDate>();
        ArrayList<LocalDate> nonBusinessDates = new ArrayList<LocalDate>();
        while (date.isBefore(endDate)) {
            CalendarDay<Instant> bs = this.calendarDay(date);
            if (bs.isBusinessDay()) {
                ++businessDays;
                businessDates.add(date);
            } else {
                ++nonBusinessDays;
                nonBusinessDates.add(date);
            }
            businessTimeNanos += bs.businessNanos();
            date = date.plusDays(1L);
        }
        return new SummaryData(key, start.toInstant(), start.toLocalDate(), end.toInstant(), end.toLocalDate(), businessTimeNanos, businessDays, nonBusinessDays, businessDates, nonBusinessDates);
    }

    private SummaryData computeMonthSummary(int key) {
        int year = YearMonthSummaryCache.yearFromYearMonthKey(key);
        int month = YearMonthSummaryCache.monthFromYearMonthKey(key);
        LocalDate startDate = LocalDate.of(year, month, 1);
        LocalDate endDate = startDate.plusMonths(1L);
        return this.summarize(key, startDate, endDate);
    }

    private SummaryData computeYearSummary(int year) {
        Instant startInstant = null;
        LocalDate startDate = null;
        Instant endInstant = null;
        LocalDate endDate = null;
        long businessTimeNanos = 0L;
        int businessDays = 0;
        int nonBusinessDays = 0;
        ArrayList<LocalDate> businessDates = new ArrayList<LocalDate>();
        ArrayList<LocalDate> nonBusinessDates = new ArrayList<LocalDate>();
        for (int month = 1; month <= 12; ++month) {
            SummaryData ms = this.summaryCache.getMonthSummary(year, month);
            if (month == 1) {
                startInstant = ms.startInstant;
                startDate = ms.startDate;
            }
            if (month == 12) {
                endInstant = ms.endInstant;
                endDate = ms.endDate;
            }
            businessTimeNanos += ms.businessTimeNanos;
            businessDays += ms.businessDays;
            nonBusinessDays += ms.nonBusinessDays;
            businessDates.addAll(ms.businessDates);
            nonBusinessDates.addAll(ms.nonBusinessDates);
        }
        return new SummaryData(year, startInstant, startDate, endInstant, endDate, businessTimeNanos, businessDays, nonBusinessDays, businessDates, nonBusinessDates);
    }

    private SummaryData getYearSummary(int year) {
        if (year < this.yearCacheStart || year > this.yearCacheEnd) {
            throw new InvalidDateException("Business calendar does not contain a complete year for: year=" + year);
        }
        return this.summaryCache.getYearSummary(year);
    }

    static int schedulesCacheKeyFromDate(LocalDate date) {
        return date.getYear() * 10000 + date.getMonthValue() * 100 + date.getDayOfMonth();
    }

    static LocalDate schedulesCacheDateFromKey(int key) {
        return LocalDate.of(key / 10000, key % 10000 / 100, key % 100);
    }

    private ReadOptimizedConcurrentCache.Pair<CalendarDay<Instant>> computeCalendarDay(int key) {
        LocalDate date = BusinessCalendar.schedulesCacheDateFromKey(key);
        CalendarDay<Instant> h = this.holidays.get(date);
        CalendarDay<Instant> v = h != null ? h : (this.weekendDays.contains(date.getDayOfWeek()) ? CalendarDay.toInstant(CalendarDay.HOLIDAY, date, this.timeZone()) : CalendarDay.toInstant(this.standardBusinessDay, date, this.timeZone()));
        return new ReadOptimizedConcurrentCache.Pair<CalendarDay<Instant>>(key, v);
    }

    public BusinessCalendar(String name, String description, ZoneId timeZone, LocalDate firstValidDate, LocalDate lastValidDate, CalendarDay<LocalTime> standardBusinessDay, Set<DayOfWeek> weekendDays, Map<LocalDate, CalendarDay<Instant>> holidays) {
        super(name, description, timeZone);
        this.firstValidDate = (LocalDate)Require.neqNull((Object)firstValidDate, (String)"firstValidDate");
        this.lastValidDate = (LocalDate)Require.neqNull((Object)lastValidDate, (String)"lastValidDate");
        this.standardBusinessDay = (CalendarDay)Require.neqNull(standardBusinessDay, (String)"standardBusinessDay");
        this.weekendDays = Set.copyOf((Collection)Require.neqNull(weekendDays, (String)"weekendDays"));
        this.holidays = Map.copyOf((Map)Require.neqNull(holidays, (String)"holidays"));
        this.yearCacheStart = firstValidDate.getDayOfYear() == 1 ? firstValidDate.getYear() : firstValidDate.getYear() + 1;
        this.yearCacheEnd = lastValidDate.isLeapYear() && lastValidDate.getDayOfYear() == 366 || lastValidDate.getDayOfYear() == 365 ? lastValidDate.getYear() : lastValidDate.getYear() - 1;
    }

    public LocalDate firstValidDate() {
        return this.firstValidDate;
    }

    public LocalDate lastValidDate() {
        return this.lastValidDate;
    }

    public Set<DayOfWeek> weekendDays() {
        return this.weekendDays;
    }

    public CalendarDay<LocalTime> standardBusinessDay() {
        return this.standardBusinessDay;
    }

    public long standardBusinessNanos() {
        return this.standardBusinessDay.businessNanos();
    }

    public Duration standardBusinessDuration() {
        return this.standardBusinessDay.businessDuration();
    }

    public Map<LocalDate, CalendarDay<Instant>> holidays() {
        return this.holidays;
    }

    public CalendarDay<Instant> calendarDay(LocalDate date) {
        if (date == null) {
            return null;
        }
        if (date.isBefore(this.firstValidDate)) {
            throw new InvalidDateException("Date is before the first valid business calendar date:  date=" + String.valueOf(date) + " firstValidDate=" + String.valueOf(this.firstValidDate));
        }
        if (date.isAfter(this.lastValidDate)) {
            throw new InvalidDateException("Date is after the last valid business calendar date:  date=" + String.valueOf(date) + " lastValidDate=" + String.valueOf(this.lastValidDate));
        }
        int key = BusinessCalendar.schedulesCacheKeyFromDate(date);
        return this.schedulesCache.computeIfAbsent(key).getValue();
    }

    public CalendarDay<Instant> calendarDay(ZonedDateTime time) {
        if (time == null) {
            return null;
        }
        return this.calendarDay(time.withZoneSameInstant(this.timeZone()).toLocalDate());
    }

    public CalendarDay<Instant> calendarDay(Instant time) {
        if (time == null) {
            return null;
        }
        return this.calendarDay(time.atZone(this.timeZone()));
    }

    public CalendarDay<Instant> calendarDay(String date) {
        if (date == null) {
            return null;
        }
        return this.calendarDay(DateTimeUtils.parseLocalDate(date));
    }

    public CalendarDay<Instant> calendarDay() {
        return this.calendarDay(this.calendarDate());
    }

    public boolean isBusinessDay(LocalDate date) {
        if (date == null) {
            return false;
        }
        return this.calendarDay(date).isBusinessDay();
    }

    public boolean isBusinessDay(String date) {
        if (date == null) {
            return false;
        }
        return this.calendarDay(date).isBusinessDay();
    }

    public boolean isBusinessDay(ZonedDateTime time) {
        if (time == null) {
            return false;
        }
        return this.calendarDay(time).isBusinessDay();
    }

    public boolean isBusinessDay(Instant time) {
        if (time == null) {
            return false;
        }
        return this.calendarDay(time).isBusinessDay();
    }

    public boolean isBusinessDay(DayOfWeek day) {
        if (day == null) {
            return false;
        }
        return !this.weekendDays.contains(day);
    }

    public boolean isBusinessDay() {
        return this.isBusinessDay(this.calendarDate());
    }

    boolean isLastBusinessDayOfMonth(LocalDate date) {
        if (date == null || !this.isBusinessDay(date)) {
            return false;
        }
        LocalDate nextBusAfterDate = this.plusBusinessDays(date, 1);
        assert (nextBusAfterDate != null);
        return date.getMonth() != nextBusAfterDate.getMonth();
    }

    public boolean isLastBusinessDayOfMonth(ZonedDateTime time) {
        if (time == null) {
            return false;
        }
        Require.neqNull((Object)time, (String)"time");
        return this.isLastBusinessDayOfMonth(DateTimeUtils.toLocalDate(time.withZoneSameInstant(this.timeZone())));
    }

    public boolean isLastBusinessDayOfMonth(Instant time) {
        if (time == null) {
            return false;
        }
        return this.isLastBusinessDayOfMonth(DateTimeUtils.toLocalDate(time, this.timeZone()));
    }

    public boolean isLastBusinessDayOfMonth(String date) {
        if (date == null) {
            return false;
        }
        return this.isLastBusinessDayOfMonth(DateTimeUtils.parseLocalDate(date));
    }

    public boolean isLastBusinessDayOfMonth() {
        return this.isLastBusinessDayOfMonth(this.calendarDate());
    }

    public boolean isLastBusinessDayOfWeek(LocalDate date) {
        if (date == null || !this.isBusinessDay(date)) {
            return false;
        }
        LocalDate nextBusinessDay = this.plusBusinessDays(date, 1);
        return date.getDayOfWeek().compareTo(nextBusinessDay.getDayOfWeek()) > 0 || this.numberCalendarDates(date, nextBusinessDay) > 6;
    }

    public boolean isLastBusinessDayOfWeek(ZonedDateTime time) {
        if (time == null) {
            return false;
        }
        return this.isLastBusinessDayOfWeek(time.withZoneSameInstant(this.timeZone()).toLocalDate());
    }

    public boolean isLastBusinessDayOfWeek(Instant time) {
        if (time == null) {
            return false;
        }
        return this.isLastBusinessDayOfWeek(DateTimeUtils.toLocalDate(time, this.timeZone()));
    }

    public boolean isLastBusinessDayOfWeek(String date) {
        if (date == null) {
            return false;
        }
        return this.isLastBusinessDayOfWeek(DateTimeUtils.parseLocalDate(date));
    }

    public boolean isLastBusinessDayOfWeek() {
        return this.isLastBusinessDayOfWeek(this.calendarDate());
    }

    boolean isLastBusinessDayOfYear(LocalDate date) {
        if (date == null || !this.isBusinessDay(date)) {
            return false;
        }
        LocalDate nextBusAfterDate = this.plusBusinessDays(date, 1);
        assert (nextBusAfterDate != null);
        return date.getYear() != nextBusAfterDate.getYear();
    }

    public boolean isLastBusinessDayOfYear(ZonedDateTime time) {
        if (time == null) {
            return false;
        }
        return this.isLastBusinessDayOfYear(DateTimeUtils.toLocalDate(time.withZoneSameInstant(this.timeZone())));
    }

    public boolean isLastBusinessDayOfYear(Instant time) {
        if (time == null) {
            return false;
        }
        return this.isLastBusinessDayOfYear(DateTimeUtils.toLocalDate(time, this.timeZone()));
    }

    boolean isLastBusinessDayOfYear(String date) {
        if (date == null) {
            return false;
        }
        Require.neqNull((Object)date, (String)"date");
        return this.isLastBusinessDayOfYear(DateTimeUtils.parseLocalDate(date));
    }

    public boolean isLastBusinessDayOfYear() {
        return this.isLastBusinessDayOfYear(this.calendarDate());
    }

    public boolean isBusinessTime(ZonedDateTime time) {
        if (time == null) {
            return false;
        }
        return this.calendarDay(time).isBusinessTime(time.toInstant());
    }

    public boolean isBusinessTime(Instant time) {
        if (time == null) {
            return false;
        }
        return this.calendarDay(time).isBusinessTime(time);
    }

    public boolean isBusinessTime() {
        return this.isBusinessTime(DateTimeUtils.now());
    }

    public double fractionStandardBusinessDay(LocalDate date) {
        if (date == null) {
            return -1.7976931348623157E308;
        }
        CalendarDay<Instant> schedule = this.calendarDay(date);
        return (double)schedule.businessNanos() / (double)this.standardBusinessNanos();
    }

    public double fractionStandardBusinessDay(String date) {
        if (date == null) {
            return -1.7976931348623157E308;
        }
        return this.fractionStandardBusinessDay(DateTimeUtils.parseLocalDate(date));
    }

    public double fractionStandardBusinessDay(Instant time) {
        if (time == null) {
            return -1.7976931348623157E308;
        }
        return this.fractionStandardBusinessDay(DateTimeUtils.toLocalDate(time, this.timeZone()));
    }

    public double fractionStandardBusinessDay(ZonedDateTime time) {
        if (time == null) {
            return -1.7976931348623157E308;
        }
        return this.fractionStandardBusinessDay(DateTimeUtils.toLocalDate(time.toInstant(), this.timeZone()));
    }

    public double fractionStandardBusinessDay() {
        return this.fractionStandardBusinessDay(this.calendarDate());
    }

    public double fractionBusinessDayComplete(Instant time) {
        if (time == null) {
            return -1.7976931348623157E308;
        }
        CalendarDay<Instant> schedule = this.calendarDay(time);
        if (!schedule.isBusinessDay()) {
            return 1.0;
        }
        long businessDaySoFar = schedule.businessNanosElapsed(time);
        return (double)businessDaySoFar / (double)schedule.businessNanos();
    }

    public double fractionBusinessDayComplete(ZonedDateTime time) {
        if (time == null) {
            return -1.7976931348623157E308;
        }
        return this.fractionBusinessDayComplete(time.toInstant());
    }

    public double fractionBusinessDayComplete() {
        return this.fractionBusinessDayComplete(DateTimeUtils.now());
    }

    public double fractionBusinessDayRemaining(Instant time) {
        if (time == null) {
            return -1.7976931348623157E308;
        }
        return 1.0 - this.fractionBusinessDayComplete(time);
    }

    public double fractionBusinessDayRemaining(ZonedDateTime time) {
        if (time == null) {
            return -1.7976931348623157E308;
        }
        return 1.0 - this.fractionBusinessDayComplete(time);
    }

    public double fractionBusinessDayRemaining() {
        return this.fractionBusinessDayRemaining(DateTimeUtils.now());
    }

    private int numberBusinessDatesInternal(LocalDate start, LocalDate end, boolean startInclusive, boolean endInclusive) {
        int days = 0;
        LocalDate day = start;
        while (!day.isAfter(end)) {
            boolean skip;
            boolean bl = skip = !startInclusive && day.equals(start) || !endInclusive && day.equals(end);
            if (!skip && this.isBusinessDay(day)) {
                ++days;
            }
            day = day.plusDays(1L);
        }
        return days;
    }

    public int numberBusinessDates(LocalDate start, LocalDate end, boolean startInclusive, boolean endInclusive) {
        if (start == null || end == null) {
            return Integer.MIN_VALUE;
        }
        if (start.isAfter(end)) {
            return 0;
        }
        SummaryData summaryFirst = null;
        SummaryData summary = null;
        int days = 0;
        Iterator<SummaryData> it = this.summaryCache.iterator(start, end, startInclusive, endInclusive);
        while (it.hasNext()) {
            summary = it.next();
            if (summaryFirst == null) {
                summaryFirst = summary;
            }
            days += summary.businessDays;
        }
        if (summaryFirst == null) {
            return this.numberBusinessDatesInternal(start, end, startInclusive, endInclusive);
        }
        days += this.numberBusinessDatesInternal(start, summaryFirst.startDate, startInclusive, false);
        return days += this.numberBusinessDatesInternal(summary.endDate, end, true, endInclusive);
    }

    public int numberBusinessDates(String start, String end, boolean startInclusive, boolean endInclusive) {
        if (start == null || end == null) {
            return Integer.MIN_VALUE;
        }
        return this.numberBusinessDates(DateTimeUtils.parseLocalDate(start), DateTimeUtils.parseLocalDate(end), startInclusive, endInclusive);
    }

    public int numberBusinessDates(ZonedDateTime start, ZonedDateTime end, boolean startInclusive, boolean endInclusive) {
        if (start == null || end == null) {
            return Integer.MIN_VALUE;
        }
        return this.numberBusinessDates(start.withZoneSameInstant(this.timeZone()).toLocalDate(), end.withZoneSameInstant(this.timeZone()).toLocalDate(), startInclusive, endInclusive);
    }

    public int numberBusinessDates(Instant start, Instant end, boolean startInclusive, boolean endInclusive) {
        if (start == null || end == null) {
            return Integer.MIN_VALUE;
        }
        return this.numberBusinessDates(DateTimeUtils.toLocalDate(start, this.timeZone()), DateTimeUtils.toLocalDate(end, this.timeZone()), startInclusive, endInclusive);
    }

    public int numberBusinessDates(LocalDate start, LocalDate end) {
        return this.numberBusinessDates(start, end, true, true);
    }

    public int numberBusinessDates(String start, String end) {
        return this.numberBusinessDates(start, end, true, true);
    }

    public int numberBusinessDates(ZonedDateTime start, ZonedDateTime end) {
        return this.numberBusinessDates(start, end, true, true);
    }

    public int numberBusinessDates(Instant start, Instant end) {
        return this.numberBusinessDates(start, end, true, true);
    }

    private int numberNonBusinessDatesInternal(LocalDate start, LocalDate end, boolean startInclusive, boolean endInclusive) {
        int days = 0;
        LocalDate day = start;
        while (!day.isAfter(end)) {
            boolean skip;
            boolean bl = skip = !startInclusive && day.equals(start) || !endInclusive && day.equals(end);
            if (!skip && !this.isBusinessDay(day)) {
                ++days;
            }
            day = day.plusDays(1L);
        }
        return days;
    }

    public int numberNonBusinessDates(LocalDate start, LocalDate end, boolean startInclusive, boolean endInclusive) {
        if (start == null || end == null) {
            return Integer.MIN_VALUE;
        }
        if (start.isAfter(end)) {
            return 0;
        }
        SummaryData summaryFirst = null;
        SummaryData summary = null;
        int days = 0;
        Iterator<SummaryData> it = this.summaryCache.iterator(start, end, startInclusive, endInclusive);
        while (it.hasNext()) {
            summary = it.next();
            if (summaryFirst == null) {
                summaryFirst = summary;
            }
            days += summary.nonBusinessDays;
        }
        if (summaryFirst == null) {
            return this.numberNonBusinessDatesInternal(start, end, startInclusive, endInclusive);
        }
        days += this.numberNonBusinessDatesInternal(start, summaryFirst.startDate, startInclusive, false);
        return days += this.numberNonBusinessDatesInternal(summary.endDate, end, true, endInclusive);
    }

    public int numberNonBusinessDates(String start, String end, boolean startInclusive, boolean endInclusive) {
        if (start == null || end == null) {
            return Integer.MIN_VALUE;
        }
        return this.numberNonBusinessDates(DateTimeUtils.parseLocalDate(start), DateTimeUtils.parseLocalDate(end), startInclusive, endInclusive);
    }

    public int numberNonBusinessDates(ZonedDateTime start, ZonedDateTime end, boolean startInclusive, boolean endInclusive) {
        if (start == null || end == null) {
            return Integer.MIN_VALUE;
        }
        return this.numberNonBusinessDates(start.withZoneSameInstant(this.timeZone()).toLocalDate(), end.withZoneSameInstant(this.timeZone()).toLocalDate(), startInclusive, endInclusive);
    }

    public int numberNonBusinessDates(Instant start, Instant end, boolean startInclusive, boolean endInclusive) {
        if (start == null || end == null) {
            return Integer.MIN_VALUE;
        }
        return this.numberNonBusinessDates(DateTimeUtils.toLocalDate(start, this.timeZone()), DateTimeUtils.toLocalDate(end, this.timeZone()), startInclusive, endInclusive);
    }

    public int numberNonBusinessDates(LocalDate start, LocalDate end) {
        return this.numberNonBusinessDates(start, end, true, true);
    }

    public int numberNonBusinessDates(String start, String end) {
        return this.numberNonBusinessDates(start, end, true, true);
    }

    public int numberNonBusinessDates(ZonedDateTime start, ZonedDateTime end) {
        return this.numberNonBusinessDates(start, end, true, true);
    }

    public int numberNonBusinessDates(Instant start, Instant end) {
        return this.numberNonBusinessDates(start, end, true, true);
    }

    private void businessDatesInternal(ArrayList<LocalDate> result, LocalDate start, LocalDate end, boolean startInclusive, boolean endInclusive) {
        LocalDate day = start;
        while (!day.isAfter(end)) {
            boolean skip;
            boolean bl = skip = !startInclusive && day.equals(start) || !endInclusive && day.equals(end);
            if (!skip && this.isBusinessDay(day)) {
                result.add(day);
            }
            day = day.plusDays(1L);
        }
    }

    public LocalDate[] businessDates(LocalDate start, LocalDate end, boolean startInclusive, boolean endInclusive) {
        if (start == null || end == null) {
            return null;
        }
        if (start.isAfter(end)) {
            return new LocalDate[0];
        }
        ArrayList<LocalDate> dateList = new ArrayList<LocalDate>();
        SummaryData summaryFirst = null;
        SummaryData summary = null;
        Iterator<SummaryData> it = this.summaryCache.iterator(start, end, startInclusive, endInclusive);
        while (it.hasNext()) {
            summary = it.next();
            if (summaryFirst == null) {
                summaryFirst = summary;
                this.businessDatesInternal(dateList, start, summaryFirst.startDate, startInclusive, false);
            }
            dateList.addAll(summary.businessDates);
        }
        if (summaryFirst == null) {
            this.businessDatesInternal(dateList, start, end, startInclusive, endInclusive);
        } else {
            this.businessDatesInternal(dateList, summary.endDate, end, true, endInclusive);
        }
        return dateList.toArray(new LocalDate[0]);
    }

    public String[] businessDates(String start, String end, boolean startInclusive, boolean endInclusive) {
        if (start == null || end == null) {
            return null;
        }
        LocalDate[] dates = this.businessDates(DateTimeUtils.parseLocalDate(start), DateTimeUtils.parseLocalDate(end), startInclusive, endInclusive);
        return dates == null ? null : (String[])Arrays.stream(dates).map(DateTimeUtils::formatDate).toArray(String[]::new);
    }

    public LocalDate[] businessDates(ZonedDateTime start, ZonedDateTime end, boolean startInclusive, boolean endInclusive) {
        if (start == null || end == null) {
            return null;
        }
        return this.businessDates(start.withZoneSameInstant(this.timeZone()).toLocalDate(), end.withZoneSameInstant(this.timeZone()).toLocalDate(), startInclusive, endInclusive);
    }

    public LocalDate[] businessDates(Instant start, Instant end, boolean startInclusive, boolean endInclusive) {
        if (start == null || end == null) {
            return null;
        }
        return this.businessDates(DateTimeUtils.toLocalDate(start, this.timeZone()), DateTimeUtils.toLocalDate(end, this.timeZone()), startInclusive, endInclusive);
    }

    public LocalDate[] businessDates(LocalDate start, LocalDate end) {
        return this.businessDates(start, end, true, true);
    }

    public String[] businessDates(String start, String end) {
        return this.businessDates(start, end, true, true);
    }

    public LocalDate[] businessDates(ZonedDateTime start, ZonedDateTime end) {
        return this.businessDates(start, end, true, true);
    }

    public LocalDate[] businessDates(Instant start, Instant end) {
        return this.businessDates(start, end, true, true);
    }

    private void nonBusinessDatesInternal(ArrayList<LocalDate> result, LocalDate start, LocalDate end, boolean startInclusive, boolean endInclusive) {
        LocalDate day = start;
        while (!day.isAfter(end)) {
            boolean skip;
            boolean bl = skip = !startInclusive && day.equals(start) || !endInclusive && day.equals(end);
            if (!skip && !this.isBusinessDay(day)) {
                result.add(day);
            }
            day = day.plusDays(1L);
        }
    }

    public LocalDate[] nonBusinessDates(LocalDate start, LocalDate end, boolean startInclusive, boolean endInclusive) {
        if (start == null || end == null) {
            return null;
        }
        if (start.isAfter(end)) {
            return new LocalDate[0];
        }
        ArrayList<LocalDate> dateList = new ArrayList<LocalDate>();
        SummaryData summaryFirst = null;
        SummaryData summary = null;
        Iterator<SummaryData> it = this.summaryCache.iterator(start, end, startInclusive, endInclusive);
        while (it.hasNext()) {
            summary = it.next();
            if (summaryFirst == null) {
                summaryFirst = summary;
                this.nonBusinessDatesInternal(dateList, start, summaryFirst.startDate, startInclusive, false);
            }
            dateList.addAll(summary.nonBusinessDates);
        }
        if (summaryFirst == null) {
            this.nonBusinessDatesInternal(dateList, start, end, startInclusive, endInclusive);
        } else {
            this.nonBusinessDatesInternal(dateList, summary.endDate, end, true, endInclusive);
        }
        return dateList.toArray(new LocalDate[0]);
    }

    public String[] nonBusinessDates(String start, String end, boolean startInclusive, boolean endInclusive) {
        if (start == null || end == null) {
            return null;
        }
        LocalDate[] dates = this.nonBusinessDates(DateTimeUtils.parseLocalDate(start), DateTimeUtils.parseLocalDate(end), startInclusive, endInclusive);
        return dates == null ? null : (String[])Arrays.stream(dates).map(DateTimeUtils::formatDate).toArray(String[]::new);
    }

    public LocalDate[] nonBusinessDates(ZonedDateTime start, ZonedDateTime end, boolean startInclusive, boolean endInclusive) {
        if (start == null || end == null) {
            return null;
        }
        return this.nonBusinessDates(start.withZoneSameInstant(this.timeZone()).toLocalDate(), end.withZoneSameInstant(this.timeZone()).toLocalDate(), startInclusive, endInclusive);
    }

    public LocalDate[] nonBusinessDates(Instant start, Instant end, boolean startInclusive, boolean endInclusive) {
        if (start == null || end == null) {
            return null;
        }
        return this.nonBusinessDates(DateTimeUtils.toLocalDate(start, this.timeZone()), DateTimeUtils.toLocalDate(end, this.timeZone()), startInclusive, endInclusive);
    }

    public LocalDate[] nonBusinessDates(LocalDate start, LocalDate end) {
        return this.nonBusinessDates(start, end, true, true);
    }

    public String[] nonBusinessDates(String start, String end) {
        return this.nonBusinessDates(start, end, true, true);
    }

    public LocalDate[] nonBusinessDates(ZonedDateTime start, ZonedDateTime end) {
        return this.nonBusinessDates(start, end, true, true);
    }

    public LocalDate[] nonBusinessDates(Instant start, Instant end) {
        return this.nonBusinessDates(start, end, true, true);
    }

    private long diffBusinessNanosInternal(Instant start, LocalDate startDate, Instant end, LocalDate endDate) {
        if (startDate.equals(endDate)) {
            CalendarDay<Instant> schedule = this.calendarDay(startDate);
            return schedule.businessNanosElapsed(end) - schedule.businessNanosElapsed(start);
        }
        long rst = this.calendarDay(startDate).businessNanosRemaining(start) + this.calendarDay(endDate).businessNanosElapsed(end);
        LocalDate d = startDate.plusDays(1L);
        while (d.isBefore(endDate)) {
            rst += this.calendarDay(d).businessNanos();
            d = d.plusDays(1L);
        }
        return rst;
    }

    public long diffBusinessNanos(Instant start, Instant end) {
        if (start == null || end == null) {
            return Long.MIN_VALUE;
        }
        if (DateTimeUtils.isAfter(start, end)) {
            return -this.diffBusinessNanos(end, start);
        }
        LocalDate startDate = DateTimeUtils.toLocalDate(start, this.timeZone());
        LocalDate endDate = DateTimeUtils.toLocalDate(end, this.timeZone());
        assert (startDate != null);
        assert (endDate != null);
        SummaryData summaryFirst = null;
        SummaryData summary = null;
        long nanos = 0L;
        Iterator<SummaryData> it = this.summaryCache.iterator(startDate, endDate, false, false);
        while (it.hasNext()) {
            summary = it.next();
            if (summaryFirst == null) {
                summaryFirst = summary;
            }
            nanos += summary.businessTimeNanos;
        }
        if (summaryFirst == null) {
            return this.diffBusinessNanosInternal(start, startDate, end, endDate);
        }
        nanos += this.diffBusinessNanosInternal(start, startDate, summaryFirst.startInstant, summaryFirst.startDate);
        return nanos += this.diffBusinessNanosInternal(summary.endInstant, summary.endDate, end, endDate);
    }

    public long diffBusinessNanos(ZonedDateTime start, ZonedDateTime end) {
        if (start == null || end == null) {
            return Long.MIN_VALUE;
        }
        return this.diffBusinessNanos(start.toInstant(), end.toInstant());
    }

    public long diffNonBusinessNanos(Instant start, Instant end) {
        if (start == null || end == null) {
            return Long.MIN_VALUE;
        }
        return DateTimeUtils.diffNanos(start, end) - this.diffBusinessNanos(start, end);
    }

    public long diffNonBusinessNanos(ZonedDateTime start, ZonedDateTime end) {
        if (start == null || end == null) {
            return Long.MIN_VALUE;
        }
        return this.diffNonBusinessNanos(start.toInstant(), end.toInstant());
    }

    public Duration diffBusinessDuration(Instant start, Instant end) {
        if (start == null || end == null) {
            return null;
        }
        return Duration.ofNanos(this.diffBusinessNanos(start, end));
    }

    public Duration diffBusinessDuration(ZonedDateTime start, ZonedDateTime end) {
        if (start == null || end == null) {
            return null;
        }
        return Duration.ofNanos(this.diffBusinessNanos(start, end));
    }

    public Duration diffNonBusinessDuration(Instant start, Instant end) {
        if (start == null || end == null) {
            return null;
        }
        return Duration.ofNanos(this.diffNonBusinessNanos(start, end));
    }

    public Duration diffNonBusinessDuration(ZonedDateTime start, ZonedDateTime end) {
        if (start == null || end == null) {
            return null;
        }
        return Duration.ofNanos(this.diffNonBusinessNanos(start, end));
    }

    public double diffBusinessDays(Instant start, Instant end) {
        if (start == null || end == null) {
            return -1.7976931348623157E308;
        }
        return (double)this.diffBusinessNanos(start, end) / (double)this.standardBusinessNanos();
    }

    public double diffBusinessDays(ZonedDateTime start, ZonedDateTime end) {
        if (start == null || end == null) {
            return -1.7976931348623157E308;
        }
        return (double)this.diffBusinessNanos(start, end) / (double)this.standardBusinessNanos();
    }

    public double diffBusinessYears(Instant start, Instant end) {
        int yearEnd;
        if (start == null || end == null) {
            return -1.7976931348623157E308;
        }
        int yearStart = DateTimeUtils.year(start, this.timeZone());
        if (yearStart == (yearEnd = DateTimeUtils.year(end, this.timeZone()))) {
            return (double)this.diffBusinessNanos(start, end) / (double)this.getYearSummary((int)yearStart).businessTimeNanos;
        }
        SummaryData yearDataStart = this.getYearSummary(yearStart);
        SummaryData yearDataEnd = this.getYearSummary(yearEnd);
        return (double)this.diffBusinessNanos(start, yearDataStart.endInstant) / (double)yearDataStart.businessTimeNanos + (double)this.diffBusinessNanos(yearDataEnd.startInstant, end) / (double)yearDataEnd.businessTimeNanos + (double)yearEnd - (double)yearStart - 1.0;
    }

    public double diffBusinessYears(ZonedDateTime start, ZonedDateTime end) {
        if (start == null || end == null) {
            return -1.7976931348623157E308;
        }
        return this.diffBusinessYears(start.toInstant(), end.toInstant());
    }

    public LocalDate plusBusinessDays(LocalDate date, int days) {
        if (date == null || days == Integer.MIN_VALUE) {
            return null;
        }
        if (days == 0) {
            return this.isBusinessDay(date) ? date : null;
        }
        int step = days > 0 ? 1 : -1;
        LocalDate d = date;
        for (int count = 0; count != days; count += this.isBusinessDay(d = d.plusDays(step)) ? step : 0) {
        }
        return d;
    }

    public String plusBusinessDays(String date, int days) {
        if (date == null || days == Integer.MIN_VALUE) {
            return null;
        }
        LocalDate d = this.plusBusinessDays(DateTimeUtils.parseLocalDate(date), days);
        return d == null ? null : d.toString();
    }

    public Instant plusBusinessDays(Instant time, int days) {
        if (time == null || days == Integer.MIN_VALUE) {
            return null;
        }
        ZonedDateTime zdt = this.plusBusinessDays(DateTimeUtils.toZonedDateTime(time, this.timeZone()), days);
        return zdt == null ? null : zdt.toInstant();
    }

    public ZonedDateTime plusBusinessDays(ZonedDateTime time, int days) {
        if (time == null || days == Integer.MIN_VALUE) {
            return null;
        }
        ZonedDateTime zdt = time.withZoneSameInstant(this.timeZone());
        LocalDate pbd = this.plusBusinessDays(zdt.toLocalDate(), days);
        return pbd == null ? null : pbd.atTime(zdt.toLocalTime()).atZone(this.timeZone());
    }

    public LocalDate minusBusinessDays(LocalDate date, int days) {
        if (date == null || days == Integer.MIN_VALUE) {
            return null;
        }
        return this.plusBusinessDays(date, -days);
    }

    public String minusBusinessDays(String date, int days) {
        if (date == null || days == Integer.MIN_VALUE) {
            return null;
        }
        return this.plusBusinessDays(date, -days);
    }

    public Instant minusBusinessDays(Instant time, int days) {
        if (time == null || days == Integer.MIN_VALUE) {
            return null;
        }
        return this.plusBusinessDays(time, -days);
    }

    public ZonedDateTime minusBusinessDays(ZonedDateTime time, int days) {
        if (time == null || days == Integer.MIN_VALUE) {
            return null;
        }
        return this.plusBusinessDays(time, -days);
    }

    public LocalDate plusNonBusinessDays(LocalDate date, int days) {
        if (date == null || days == Integer.MIN_VALUE) {
            return null;
        }
        if (days == 0) {
            return this.isBusinessDay(date) ? null : date;
        }
        int step = days > 0 ? 1 : -1;
        LocalDate d = date;
        for (int count = 0; count != days; count += this.isBusinessDay(d = d.plusDays(step)) ? 0 : step) {
        }
        return d;
    }

    public String plusNonBusinessDays(String date, int days) {
        if (date == null || days == Integer.MIN_VALUE) {
            return null;
        }
        LocalDate d = this.plusNonBusinessDays(DateTimeUtils.parseLocalDate(date), days);
        return d == null ? null : d.toString();
    }

    public Instant plusNonBusinessDays(Instant time, int days) {
        if (time == null || days == Integer.MIN_VALUE) {
            return null;
        }
        ZonedDateTime zdt = this.plusNonBusinessDays(DateTimeUtils.toZonedDateTime(time, this.timeZone()), days);
        return zdt == null ? null : zdt.toInstant();
    }

    public ZonedDateTime plusNonBusinessDays(ZonedDateTime time, int days) {
        if (time == null || days == Integer.MIN_VALUE) {
            return null;
        }
        ZonedDateTime zdt = time.withZoneSameInstant(this.timeZone());
        LocalDate pbd = this.plusNonBusinessDays(zdt.toLocalDate(), days);
        return pbd == null ? null : pbd.atTime(zdt.toLocalTime()).atZone(this.timeZone());
    }

    public LocalDate minusNonBusinessDays(LocalDate date, int days) {
        if (date == null || days == Integer.MIN_VALUE) {
            return null;
        }
        return this.plusNonBusinessDays(date, -days);
    }

    public String minusNonBusinessDays(String date, int days) {
        if (date == null || days == Integer.MIN_VALUE) {
            return null;
        }
        return this.plusNonBusinessDays(date, -days);
    }

    public Instant minusNonBusinessDays(Instant time, int days) {
        if (time == null || days == Integer.MIN_VALUE) {
            return null;
        }
        return this.plusNonBusinessDays(time, -days);
    }

    public ZonedDateTime minusNonBusinessDays(ZonedDateTime time, int days) {
        if (time == null || days == Integer.MIN_VALUE) {
            return null;
        }
        return this.plusNonBusinessDays(time, -days);
    }

    public LocalDate futureBusinessDate(int days) {
        if (days == Integer.MIN_VALUE) {
            return null;
        }
        return this.plusBusinessDays(this.calendarDate(), days);
    }

    public LocalDate pastBusinessDate(int days) {
        if (days == Integer.MIN_VALUE) {
            return null;
        }
        return this.minusBusinessDays(this.calendarDate(), days);
    }

    public LocalDate futureNonBusinessDate(int days) {
        if (days == Integer.MIN_VALUE) {
            return null;
        }
        return this.plusNonBusinessDays(this.calendarDate(), days);
    }

    public LocalDate pastNonBusinessDate(int days) {
        if (days == Integer.MIN_VALUE) {
            return null;
        }
        return this.minusNonBusinessDays(this.calendarDate(), days);
    }

    private static class SummaryData
    extends ReadOptimizedConcurrentCache.IntKeyedValue {
        private final Instant startInstant;
        private final LocalDate startDate;
        private final Instant endInstant;
        private final LocalDate endDate;
        private final long businessTimeNanos;
        private final int businessDays;
        private final int nonBusinessDays;
        private final ArrayList<LocalDate> businessDates;
        private final ArrayList<LocalDate> nonBusinessDates;

        public SummaryData(int key, Instant startInstant, LocalDate startDate, Instant endInstant, LocalDate endDate, long businessTimeNanos, int businessDays, int nonBusinessDays, ArrayList<LocalDate> businessDates, ArrayList<LocalDate> nonBusinessDates) {
            super(key);
            this.startInstant = startInstant;
            this.startDate = startDate;
            this.endInstant = endInstant;
            this.endDate = endDate;
            this.businessTimeNanos = businessTimeNanos;
            this.businessDays = businessDays;
            this.nonBusinessDays = nonBusinessDays;
            this.businessDates = businessDates;
            this.nonBusinessDates = nonBusinessDates;
        }
    }

    public static class InvalidDateException
    extends RuntimeException {
        private InvalidDateException(String message) {
            super(message);
        }

        public InvalidDateException(String message, Throwable cause) {
            super(message, cause);
        }
    }
}

