/*
 * Decompiled with CFR 0.152.
 */
package io.cryostat.agent.harvest;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.cryostat.agent.CryostatClient;
import io.cryostat.agent.FlightRecorderHelper;
import io.cryostat.agent.Registration;
import io.cryostat.agent.util.StringUtils;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.UnaryOperator;
import jdk.jfr.FlightRecorder;
import jdk.jfr.FlightRecorderListener;
import jdk.jfr.Recording;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Harvester
implements FlightRecorderListener {
    public static final String RECORDING_NAME_ON_EXIT = "onexit";
    public static final String RECORDING_NAME_PERIODIC = "cryostat-agent-harvester";
    private static final String AUTOANALYZE_LABEL = "autoanalyze";
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final ScheduledExecutorService executor;
    private final ScheduledExecutorService workerPool;
    private final long period;
    private final String template;
    private final int maxFiles;
    private final RecordingSettings exitSettings;
    private final RecordingSettings periodicSettings;
    private final boolean autoanalyze;
    private final CryostatClient client;
    private final FlightRecorderHelper flightRecorderHelper;
    private final Set<FlightRecorderHelper.TemplatedRecording> recordings = ConcurrentHashMap.newKeySet();
    private Optional<FlightRecorderHelper.TemplatedRecording> sownRecording = Optional.empty();
    private final Map<FlightRecorderHelper.TemplatedRecording, Path> exitPaths = new ConcurrentHashMap<FlightRecorderHelper.TemplatedRecording, Path>();
    private final AtomicBoolean exitUploadInitiated = new AtomicBoolean(false);
    private FlightRecorder flightRecorder;
    private Future<?> task;
    private boolean running;

    @SuppressFBWarnings(value={"EI_EXPOSE_REP2"})
    public Harvester(ScheduledExecutorService executor, ScheduledExecutorService workerPool, long period, String template, int maxFiles, RecordingSettings exitSettings, RecordingSettings periodicSettings, boolean autoanalyze, CryostatClient client, FlightRecorderHelper flightRecorderHelper, Registration registration) {
        this.executor = executor;
        this.workerPool = workerPool;
        this.period = period;
        this.template = template;
        this.maxFiles = maxFiles;
        this.exitSettings = exitSettings;
        this.periodicSettings = periodicSettings;
        this.autoanalyze = autoanalyze;
        this.client = client;
        this.flightRecorderHelper = flightRecorderHelper;
        registration.addRegistrationListener(evt -> {
            switch (evt.state) {
                case REGISTERED: {
                    break;
                }
                case UNREGISTERED: {
                    executor.submit(this::stop);
                    break;
                }
                case REFRESHED: {
                    break;
                }
                case PUBLISHED: {
                    executor.submit(this::start);
                    break;
                }
            }
        });
    }

    public void start() {
        this.executor.submit(() -> {
            if (this.running) {
                return;
            }
            this.running = true;
            if (StringUtils.isBlank(this.template)) {
                this.log.warn("Template not specified");
            }
            if (this.maxFiles <= 0) {
                this.log.warn("Maximum number of files to keep within target is {} <= 0", (Object)this.maxFiles);
            }
            if (!FlightRecorder.isAvailable()) {
                this.log.error("FlightRecorder is unavailable");
                return;
            }
            this.log.debug("JFR Harvester starting");
            try {
                FlightRecorder.addListener(this);
                this.flightRecorder = FlightRecorder.getFlightRecorder();
                if (this.exitSettings.maxAge > 0L) {
                    this.log.debug("On-stop uploads will contain approximately the most recent {}ms ({}) of data", (Object)this.exitSettings.maxAge, (Object)Duration.ofMillis(this.exitSettings.maxAge));
                }
                if (this.exitSettings.maxSize > 0L) {
                    this.log.debug("On-stop uploads will contain approximately the most recent {} bytes ({}) of data", (Object)this.exitSettings.maxSize, (Object)FileUtils.byteCountToDisplaySize((long)this.exitSettings.maxSize));
                }
                if (this.periodicSettings.maxAge > 0L) {
                    this.log.debug("Periodic uploads will contain approximately the most recent {}ms ({}) of data", (Object)this.periodicSettings.maxAge, (Object)Duration.ofMillis(this.periodicSettings.maxAge));
                }
                if (this.periodicSettings.maxSize > 0L) {
                    this.log.debug("On-stop uploads will contain approximately the most recent {} bytes ({}) of data", (Object)this.periodicSettings.maxSize, (Object)FileUtils.byteCountToDisplaySize((long)this.periodicSettings.maxSize));
                }
            }
            catch (IllegalStateException | SecurityException e) {
                this.log.error("Harvester could not start", (Throwable)e);
                return;
            }
            this.startRecording(true);
            if (this.task != null) {
                this.task.cancel(true);
            }
            if (this.period > 0L) {
                this.log.debug("JFR Harvester started with period {}", (Object)Duration.ofMillis(this.period));
                this.task = this.workerPool.scheduleAtFixedRate(this::uploadOngoing, this.period, this.period, TimeUnit.MILLISECONDS);
            } else {
                this.log.debug("JFR Harvester started, periodic uploads disabled (period {} < 0)", (Object)this.period);
            }
        });
    }

    public void stop() {
        this.executor.submit(() -> {
            if (!this.running) {
                return;
            }
            this.log.trace("Harvester stopping");
            if (this.task != null) {
                this.task.cancel(true);
                this.task = null;
            }
            FlightRecorder.removeListener(this);
            this.log.trace("Harvester stopped");
            this.running = false;
        });
    }

    @Override
    public void recordingStateChanged(Recording recording) {
        this.log.debug("{}({}) {}", new Object[]{recording.getName(), recording.getId(), recording.getState().name()});
        this.getTrackedRecordingById(recording.getId()).ifPresent(tr -> {
            boolean isSownRecording = this.sownRecording.map(FlightRecorderHelper.TemplatedRecording::getRecording).map(Recording::getId).map(id -> id.longValue() == recording.getId()).orElse(false);
            switch (recording.getState()) {
                case NEW: {
                    break;
                }
                case DELAYED: {
                    break;
                }
                case RUNNING: {
                    break;
                }
                case STOPPED: {
                    try {
                        if (!this.exitUploadInitiated.get()) {
                            tr.getRecording().dump(this.exitPaths.get(tr));
                            this.uploadRecording((FlightRecorderHelper.TemplatedRecording)tr).get();
                        }
                    }
                    catch (IOException e) {
                        this.log.error("Failed to dump recording to file", (Throwable)e);
                    }
                    catch (InterruptedException | ExecutionException e) {
                        this.log.warn("Could not upload exit dump file", (Throwable)e);
                    }
                    if (!isSownRecording) break;
                    this.safeCloseCurrentRecording();
                    break;
                }
                case CLOSED: {
                    this.executor.submit(() -> {
                        try {
                            this.recordings.remove(tr);
                            Path exitPath = this.exitPaths.remove(tr);
                            Files.deleteIfExists(exitPath);
                            this.log.trace("Deleted temp file {}", (Object)exitPath);
                        }
                        catch (IOException e) {
                            this.log.warn("Could not delete temp file", (Throwable)e);
                        }
                        finally {
                            this.startRecording(false);
                        }
                    });
                    break;
                }
                default: {
                    this.log.warn("Unknown state {} for recording with ID {}", (Object)recording.getState(), (Object)recording.getId());
                }
            }
        });
    }

    public Future<Void> exitUpload() {
        return CompletableFuture.supplyAsync(() -> {
            this.running = false;
            if (this.flightRecorder == null) {
                return null;
            }
            try {
                if (!this.exitUploadInitiated.getAndSet(true)) {
                    this.uploadOngoing(PushType.ON_EXIT, this.exitSettings).get();
                }
            }
            catch (InterruptedException | ExecutionException e) {
                this.log.error("Exit upload failed", (Throwable)e);
                throw new CompletionException(e);
            }
            finally {
                this.safeCloseCurrentRecording();
            }
            this.log.trace("Harvester stopped");
            return null;
        }, this.executor);
    }

    public void handleNewRecording(FlightRecorderHelper.TemplatedRecording tr) {
        this.handleNewRecording(tr, this.periodicSettings);
    }

    public void handleNewRecording(FlightRecorderHelper.TemplatedRecording tr, RecordingSettings settings) {
        try {
            Recording recording = tr.getRecording();
            recording.setToDisk(true);
            recording.setDumpOnExit(true);
            recording = settings.apply(recording);
            Path path = Files.createTempFile(null, null, new FileAttribute[0]);
            Files.write(path, new byte[0], StandardOpenOption.TRUNCATE_EXISTING);
            recording.setDestination(path);
            this.log.trace("{}({}) will dump to {}", new Object[]{recording.getName(), recording.getId(), path});
            this.recordings.add(tr);
            this.exitPaths.put(tr, path);
        }
        catch (IOException ioe) {
            this.log.error("Unable to handle recording", (Throwable)ioe);
            tr.getRecording().close();
        }
    }

    private void startRecording(boolean restart) {
        this.executor.submit(() -> {
            if (StringUtils.isBlank(this.template)) {
                return;
            }
            if (restart) {
                this.safeCloseCurrentRecording();
            } else if (this.sownRecording.isPresent()) {
                return;
            }
            this.flightRecorderHelper.createRecordingWithPredefinedTemplate(this.template).ifPresent(recording -> {
                this.handleNewRecording((FlightRecorderHelper.TemplatedRecording)recording, this.periodicSettings);
                this.sownRecording = Optional.of(recording);
                recording.getRecording().start();
                this.log.debug("JFR Harvester started recording using template \"{}\"", (Object)this.template);
            });
        });
    }

    private void safeCloseCurrentRecording() {
        this.sownRecording.map(FlightRecorderHelper.TemplatedRecording::getRecording).ifPresent(Recording::close);
        this.sownRecording = Optional.empty();
    }

    private Optional<FlightRecorderHelper.TemplatedRecording> getTrackedRecordingById(long id) {
        if (id < 0L) {
            return Optional.empty();
        }
        for (FlightRecorderHelper.TemplatedRecording recording : this.recordings) {
            if (id != recording.getRecording().getId()) continue;
            return Optional.of(recording);
        }
        return Optional.empty();
    }

    private Future<Void> uploadOngoing() {
        return this.uploadOngoing(PushType.SCHEDULED, this.periodicSettings);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Future<Void> uploadOngoing(PushType pushType, RecordingSettings settings) {
        try (Recording recording = settings.apply(this.flightRecorder.takeSnapshot());){
            Path exitPath = Files.createTempFile(null, null, new FileAttribute[0]);
            Files.write(exitPath, new byte[0], StandardOpenOption.TRUNCATE_EXISTING);
            recording.dump(exitPath);
            if (Files.size(exitPath) < 1L) {
                CompletableFuture<Void> completableFuture = CompletableFuture.failedFuture(new IllegalStateException("No source recording data"));
                return completableFuture;
            }
            this.log.trace("Dumping {}({}) to {}", new Object[]{recording.getName(), recording.getId(), exitPath});
            CompletionStage completionStage = this.client.upload(pushType, this.sownRecording, this.maxFiles, this.additionalLabels(), exitPath).thenRun(() -> {
                try {
                    Files.deleteIfExists(exitPath);
                    this.log.trace("Deleted temp file {}", (Object)exitPath);
                }
                catch (IOException ioe) {
                    this.log.error("Failed to clean up snapshot dump file", (Throwable)ioe);
                }
            });
            return completionStage;
        }
    }

    private Future<Void> uploadRecording(FlightRecorderHelper.TemplatedRecording tr) throws IOException {
        Path exitPath = this.exitPaths.get(tr);
        return this.client.upload(PushType.ON_STOP, Optional.of(tr), this.maxFiles, this.additionalLabels(), exitPath);
    }

    private Map<String, String> additionalLabels() {
        HashMap<String, String> map = new HashMap<String, String>();
        if (this.autoanalyze) {
            map.put(AUTOANALYZE_LABEL, String.valueOf(true));
        }
        return map;
    }

    public static class RecordingSettings
    implements UnaryOperator<Recording> {
        public String name;
        public long maxSize;
        public long maxAge;

        @Override
        public Recording apply(Recording r) {
            if (StringUtils.isNotBlank(this.name)) {
                r.setName(this.name);
            }
            if (this.maxSize > 0L) {
                r.setMaxSize(this.maxSize);
            }
            if (this.maxAge > 0L) {
                r.setMaxAge(Duration.ofMillis(this.maxAge));
            }
            return r;
        }
    }

    public static enum PushType {
        SCHEDULED,
        ON_STOP,
        ON_EXIT;

    }
}

