/*
 * Decompiled with CFR 0.152.
 */
package com.redhat.insights.jars;

import com.redhat.insights.jars.JarInfo;
import com.redhat.insights.jars.JarUtils;
import com.redhat.insights.logging.InsightsLogger;
import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarInputStream;
import java.util.jar.Manifest;

public final class JarAnalyzer {
    public static final String SHA1_CHECKSUM_KEY = "sha1Checksum";
    public static final String SHA256_CHECKSUM_KEY = "sha256Checksum";
    public static final String SHA512_CHECKSUM_KEY = "sha512Checksum";
    private static final String JAR_EXTENSION = ".jar";
    static final String UNKNOWN_VERSION = " ";
    private static final String[] ATTRIBUTES_TO_COLLECT = new String[]{Attributes.Name.IMPLEMENTATION_VENDOR.toString(), Attributes.Name.IMPLEMENTATION_VENDOR_ID.toString()};
    private final InsightsLogger logger;
    private final boolean skipTempJars;
    private final List<String> ignoreJars;
    private static final File TEMP_DIRECTORY = new File(System.getProperty("java.io.tmpdir"));

    public JarAnalyzer(InsightsLogger logger2, boolean skipTempJars) {
        this.logger = logger2;
        this.skipTempJars = skipTempJars;
        if (!skipTempJars) {
            logger2.debug("Temporary jars will be transmitted to the host");
        }
        this.ignoreJars = new ArrayList<String>();
    }

    public Optional<JarInfo> process(URL url) throws URISyntaxException {
        String jarFile = this.parseJarName(url);
        return this.process(jarFile, url);
    }

    public Optional<JarInfo> process(String file, URL url) throws URISyntaxException {
        if (this.skipTempJars && JarAnalyzer.isTempFile(url)) {
            this.logger.debug(url + " Skipping temp jar file");
            return Optional.empty();
        }
        if (this.isModularJdkJar(url)) {
            this.logger.debug(url + " Skipping JDK jar file");
            return Optional.empty();
        }
        String archive = file.toLowerCase(Locale.ROOT);
        if (!(archive.endsWith(".zip") || archive.endsWith(JAR_EXTENSION) || archive.endsWith(".war") || archive.endsWith(".rar") || archive.endsWith(".ear") || "content".equals(archive))) {
            this.logger.debug(url + " Skipping file with non-jar extension");
            return Optional.empty();
        }
        if (this.shouldAttemptAdd(file)) {
            this.logger.debug(url + "  Adding the file " + archive + " with version");
            return Optional.of(this.getJarInfoSafe(file, url));
        }
        return Optional.empty();
    }

    static boolean isTempFile(URL address) throws URISyntaxException {
        if (!"file".equals(address.getProtocol())) {
            return false;
        }
        return JarAnalyzer.isTempFile(new File(address.toURI()));
    }

    static boolean isTempFile(File fileParam) {
        File file = fileParam;
        while (file != null) {
            if (!TEMP_DIRECTORY.equals(file = file.getParentFile())) continue;
            return true;
        }
        return false;
    }

    JarInfo getJarInfoSafe(String jarFile, URL url) {
        JarInfo jarInfo;
        HashMap<String, String> attributes = new HashMap<String, String>();
        this.setChecksumsPerf(attributes, url);
        try {
            jarInfo = this.getJarInfo(jarFile, url, attributes);
        }
        catch (Exception e) {
            this.logger.debug(url + " Trouble getting version from jar: adding jar without version");
            jarInfo = new JarInfo(jarFile, UNKNOWN_VERSION, attributes);
        }
        return jarInfo;
    }

    void setChecksumsPerf(Map<String, String> attributes, URL url) {
        try {
            String[] shaChecksums = JarUtils.computeSha(url);
            attributes.put(SHA1_CHECKSUM_KEY, shaChecksums[0]);
            attributes.put(SHA256_CHECKSUM_KEY, shaChecksums[1]);
            attributes.put(SHA512_CHECKSUM_KEY, shaChecksums[2]);
        }
        catch (Exception ex) {
            this.logger.error(url + " Error getting jar file sha checksum", ex);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private JarInfo getJarInfo(String jarFilename, URL url, Map<String, String> attributes) throws IOException {
        JarInputStream jarInputStream;
        block7: {
            JarInfo jarInfo;
            jarInputStream = JarUtils.getJarInputStream(url);
            try {
                JarAnalyzer.getExtraAttributes(jarInputStream, attributes);
                Optional<Map<String, String>> oPom = JarAnalyzer.getPom(jarInputStream);
                if (!oPom.isPresent()) break block7;
                Map<String, String> pom = oPom.get();
                attributes.putAll(pom);
                jarInfo = new JarInfo(jarFilename, pom.getOrDefault("version", UNKNOWN_VERSION), attributes);
                if (jarInputStream == null) return jarInfo;
            }
            catch (Exception ex) {
                this.logger.error(url + "Exception getting extra attributes or pom", ex);
                break block7;
            }
            catch (Throwable throwable) {
                throw throwable;
            }
            jarInputStream.close();
            return jarInfo;
        }
        Optional<String> oVersion = JarAnalyzer.getVersion(jarInputStream);
        String version = oVersion.orElse(UNKNOWN_VERSION);
        return new JarInfo(jarFilename, version, attributes);
    }

    private static Optional<Map<String, String>> getPom(JarInputStream jarFile) throws IOException {
        Optional<Map<String, String>> out = Optional.empty();
        JarEntry entry = jarFile.getNextJarEntry();
        while (entry != null) {
            if (entry.getName().startsWith("META-INF/maven") && entry.getName().endsWith("pom.properties")) {
                if (out.isPresent()) {
                    return Optional.empty();
                }
                Properties props = new Properties();
                props.load(jarFile);
                out = Optional.of(props);
            }
            entry = jarFile.getNextJarEntry();
        }
        return out;
    }

    static void getExtraAttributes(JarInputStream jarFile, Map<String, String> map) {
        Manifest manifest = jarFile.getManifest();
        if (manifest == null) {
            return;
        }
        Attributes attributes = manifest.getMainAttributes();
        for (String name : ATTRIBUTES_TO_COLLECT) {
            String value = attributes.getValue(name);
            if (null == value) continue;
            map.put(name, value);
        }
    }

    static Optional<String> getVersion(JarInputStream jarFile) {
        Manifest manifest = jarFile.getManifest();
        if (manifest == null) {
            return Optional.empty();
        }
        return Optional.of(JarUtils.getVersionFromManifest(manifest));
    }

    String parseJarName(URL url) throws URISyntaxException {
        if (this.isModularJdkJar(url)) {
            return "";
        }
        if ("file".equals(url.getProtocol())) {
            File file = new File(url.toURI());
            return file.getName().trim();
        }
        this.logger.debug("Parsing jar file name " + url);
        String path = url.getFile();
        int end = path.lastIndexOf(JAR_EXTENSION);
        if (end > 0) {
            int start = (path = path.substring(0, end)).lastIndexOf(File.separator);
            if (start > -1) {
                return path.substring(start + 1) + JAR_EXTENSION;
            }
            return path + JAR_EXTENSION;
        }
        throw new URISyntaxException(url.getPath(), "Unable to parse the jar file name from a URL");
    }

    private boolean isModularJdkJar(URL url) {
        return "jrt".equals(url.getProtocol());
    }

    private boolean shouldAttemptAdd(String jarFile) {
        String agentJarFile = Paths.get(JarAnalyzer.class.getProtectionDomain().getCodeSource().getLocation().getFile(), new String[0]).getFileName().toString();
        return !this.ignoreJars.contains(jarFile) && !jarFile.equals(agentJarFile);
    }
}

