package io.contek.tusk.log4j;

import com.google.common.collect.ImmutableMap;
import io.contek.tusk.BatchingConfig;
import io.contek.tusk.Table;
import io.contek.tusk.TuskUtils;
import org.apache.logging.log4j.Level;

import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;
import javax.annotation.concurrent.NotThreadSafe;
import java.time.Duration;
import java.util.HashMap;
import java.util.Map;

import static com.google.common.base.Preconditions.checkNotNull;

@Immutable
public final class TuskLoggingConfig {

  private final String appName;
  private final Table table;
  private final BatchingConfig batching;
  private final String appColumn;
  private final String levelColumn;
  private final String loggerColumn;
  private final String lineColumn;
  private final String errorColumn;
  private final String messageColumn;
  private final String stacktraceColumn;
  private final Level logLevel;
  private final ImmutableMap<String, String> contextColumns;
  private final ILevelMapper mapper;

  private TuskLoggingConfig(
      String appName,
      Table table,
      BatchingConfig batching,
      String appColumn,
      String levelColumn,
      @Nullable String loggerColumn,
      @Nullable String lineColumn,
      @Nullable String errorColumn,
      @Nullable String messageColumn,
      @Nullable String stacktraceColumn,
      @Nullable Level logLevel,
      ImmutableMap<String, String> contextColumns,
      ILevelMapper mapper) {
    this.appName = appName;
    this.table = table;
    this.batching = batching;
    this.levelColumn = levelColumn;
    this.appColumn = appColumn;
    this.loggerColumn = loggerColumn;
    this.lineColumn = lineColumn;
    this.errorColumn = errorColumn;
    this.messageColumn = messageColumn;
    this.stacktraceColumn = stacktraceColumn;
    this.logLevel = logLevel;
    this.contextColumns = contextColumns;
    this.mapper = mapper;
  }

  public static Builder newBuilder() {
    return new Builder();
  }

  public String getAppName() {
    return appName;
  }

  public Table getTable() {
    return table;
  }

  public BatchingConfig getBatching() {
    return batching;
  }

  public String getAppColumn() {
    return appColumn;
  }

  public String getLevelColumn() {
    return levelColumn;
  }

  @Nullable
  public String getLoggerColumn() {
    return loggerColumn;
  }

  @Nullable
  public String getLineColumn() {
    return lineColumn;
  }

  @Nullable
  public String getErrorColumn() {
    return errorColumn;
  }

  @Nullable
  public String getMessageColumn() {
    return messageColumn;
  }

  @Nullable
  public String getStacktraceColumn() {
    return stacktraceColumn;
  }

  @Nullable
  public Level getLogLevel() {
    return logLevel;
  }

  public ImmutableMap<String, String> getContextColumns() {
    return contextColumns;
  }

  public ILevelMapper getMapper() {
    return mapper;
  }

  @NotThreadSafe
  public static final class Builder {

    private String appName;
    private Table table;
    private BatchingConfig batching = BatchingConfig.forDuration(Duration.ofSeconds(10), 50000);
    private String appColumn = "app";
    private String levelColumn = "level";
    private String loggerColumn = "logger";
    private String lineColumn = "line";
    private String errorColumn = "error";
    private String messageColumn = "message";
    private String stacktraceColumn = "stacktrace";
    private String logLevel = "INFO";
    private Map<String, String> contextColumns = new HashMap<>();
    private ILevelMapper levelMapper = ILevelMapper.getDefault();

    public Builder setAppName(String appName) {
      this.appName = appName;
      return this;
    }

    public Builder setTable(Table.Builder table) {
      return setTable(table.build());
    }

    public Builder setTable(Table table) {
      this.table = table;
      return this;
    }

    public Builder setBatching(BatchingConfig batching) {
      this.batching = batching;
      return this;
    }

    public Builder setAppColumn(String appColumn) {
      this.appColumn = appColumn;
      return this;
    }

    public Builder setLevelColumn(String levelColumn) {
      this.levelColumn = levelColumn;
      return this;
    }

    public Builder setLoggerColumn(@Nullable String loggerColumn) {
      this.loggerColumn = loggerColumn;
      return this;
    }

    public Builder setLineColumn(@Nullable String lineColumn) {
      this.lineColumn = lineColumn;
      return this;
    }

    public Builder setErrorColumn(@Nullable String errorColumn) {
      this.errorColumn = errorColumn;
      return this;
    }

    public Builder setMessageColumn(@Nullable String messageColumn) {
      this.messageColumn = messageColumn;
      return this;
    }

    public Builder setStacktraceColumn(@Nullable String stacktraceColumn) {
      this.stacktraceColumn = stacktraceColumn;
      return this;
    }

    public Builder addContextColumn(String keyAndColumn) {
      return addContextColumn(keyAndColumn, keyAndColumn);
    }

    public Builder addContextColumn(String key, String column) {
      this.contextColumns.put(key, column);
      return this;
    }

    public Builder addContextColumns(Map<String, String> contextColumns) {
      this.contextColumns.putAll(contextColumns);
      return this;
    }

    public Builder setContextColumns(Map<String, String> contextColumns) {
      this.contextColumns = contextColumns;
      return this;
    }

    public Builder setLogLevel(@Nullable String logLevel) {
      this.logLevel = logLevel;
      return this;
    }

    public Builder setLevelMapper(ILevelMapper levelMapper) {
      this.levelMapper = levelMapper;
      return this;
    }

    public TuskLoggingConfig build() {
      checkNotNull(table);
      checkNotNull(batching);
      checkNotNull(levelColumn);
      checkNotNull(appColumn);
      checkNotNull(contextColumns);
      checkNotNull(levelMapper);
      if (appName == null) {
        appName = TuskUtils.getAppName(appColumn);
      }

      return new TuskLoggingConfig(
          appName,
          table,
          batching,
          appColumn,
          levelColumn,
          loggerColumn,
          lineColumn,
          errorColumn,
          messageColumn,
          stacktraceColumn,
          logLevel == null ? null : Level.getLevel(logLevel),
          ImmutableMap.copyOf(contextColumns),
          levelMapper);
    }

    private Builder() {}
  }
}
