package io.contek.tusk.log4j;

import com.google.common.base.Throwables;
import io.contek.tusk.Metric;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.core.Filter;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.appender.AbstractAppender;
import org.apache.logging.log4j.core.config.Property;
import org.apache.logging.log4j.core.layout.MessageLayout;
import org.apache.logging.log4j.util.ReadOnlyStringMap;

import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;
import java.util.Map;

@Immutable
public final class TuskAppender extends AbstractAppender {

  private final Metric metric;
  private final TuskLoggingConfig config;

  TuskAppender(
      TuskLoggingConfig config,
      String appenderName,
      @Nullable Filter filter,
      boolean ignoreExceptions,
      @Nullable Property[] properties) {
    super(appenderName, filter, new MessageLayout(), ignoreExceptions, properties);
    this.metric = Metric.metric(config.getTable(), config.getBatching());
    this.config = config;
  }

  @Override
  public void append(LogEvent event) {
    Metric.EntryWriter writer = metric.newEntry();
    Level level = event.getLevel();
    Level logLevel = config.getLogLevel();
    if (logLevel != null && level.intLevel() > logLevel.intLevel()) {
      return;
    }

    writer.putInt8(config.getLevelColumn(), config.getMapper().apply(level));
    writer.putString(config.getAppColumn(), config.getAppName());

    if (config.getLoggerColumn() != null) {
      String name = event.getLoggerName();
      String value = name == null ? "" : name;
      writer.putString(config.getLoggerColumn(), value);
    }
    if (config.getLineColumn() != null) {
      Throwable t = event.getThrown();
      StackTraceElement[] stack = t == null ? null : t.getStackTrace();
      int value = stack == null || stack.length < 1 ? 0 : stack[0].getLineNumber();
      writer.putUInt32(config.getLineColumn(), value);
    }
    if (config.getErrorColumn() != null) {
      Throwable t = event.getThrown();
      String value = t == null ? "" : t.getClass().getSimpleName();
      writer.putString(config.getErrorColumn(), value);
    }
    if (config.getMessageColumn() != null) {
      String message = event.getMessage().getFormattedMessage();
      String value = message == null ? "" : message;
      writer.putString(config.getMessageColumn(), value);
    }
    if (config.getStacktraceColumn() != null) {
      Throwable t = event.getThrown();
      String value = t == null ? "" : Throwables.getStackTraceAsString(t);
      writer.putString(config.getStacktraceColumn(), value);
    }
    ReadOnlyStringMap context = event.getContextData();
    for (Map.Entry<String, String> contextTag : context.toMap().entrySet()) {
      String contextKey = contextTag.getKey();
      String contextValue = contextTag.getValue();
      String mappedColumnName = config.getContextColumns().get(contextKey);
      if (mappedColumnName == null) {
        mappedColumnName = contextKey;
      }
      writer.putString(mappedColumnName, contextValue);
    }
    writer.write();
  }
}
