package io.contek.tusk.log4j;

import io.contek.tusk.Table;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.Filter;
import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.config.Configuration;
import org.apache.logging.log4j.core.config.LoggerConfig;
import org.apache.logging.log4j.core.config.Property;

import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;
import javax.annotation.concurrent.NotThreadSafe;

import static com.google.common.base.Preconditions.checkNotNull;

@Immutable
public final class TuskLogging {

  public static void init(String table) {
    init(null, table);
  }

  public static void init(@Nullable String database, String table) {
    init(Table.newBuilder().setDatabase(database).setName(table).build());
  }

  public static void init(Table table) {
    init(TuskLoggingConfig.newBuilder().setTable(table));
  }

  public static void init(TuskLoggingConfig.Builder config) {
    init(config.build());
  }

  public static void init(TuskLoggingConfig config) {
    customize().setConfig(config).complete();
  }

  public static Customizer customize() {
    return new Customizer();
  }

  private static void attach(TuskAppender appender) {
    LoggerContext loggerContext = (LoggerContext) LogManager.getContext(false);
    Configuration config = loggerContext.getConfiguration();
    LoggerConfig root = config.getRootLogger();
    appender.start();
    root.addAppender(appender, null, null);
    loggerContext.updateLoggers();
  }

  private TuskLogging() {}

  @NotThreadSafe
  public static final class Customizer {

    private TuskLoggingConfig config;
    private String appenderName = "Tusk";
    private Filter filter = null;
    private boolean ignoreExceptions = true;
    private Property[] properties = null;

    public Customizer setConfig(TuskLoggingConfig.Builder config) {
      return setConfig(config.build());
    }

    public Customizer setConfig(TuskLoggingConfig config) {
      this.config = config;
      return this;
    }

    public Customizer setAppenderName(String appenderName) {
      this.appenderName = appenderName;
      return this;
    }

    public Customizer setFilter(@Nullable Filter filter) {
      this.filter = filter;
      return this;
    }

    public Customizer setIgnoreExceptions(boolean ignoreExceptions) {
      this.ignoreExceptions = ignoreExceptions;
      return this;
    }

    public Customizer setProperties(@Nullable Property[] properties) {
      this.properties = properties;
      return this;
    }

    public void complete() {
      checkNotNull(config);
      checkNotNull(appenderName);

      attach(new TuskAppender(config, appenderName, filter, ignoreExceptions, properties));
    }

    private Customizer() {}
  }
}
