package io.contek.tusk.heartbeat;

import io.contek.tusk.BatchingConfig;
import io.contek.tusk.Table;
import io.contek.tusk.TuskUtils;

import javax.annotation.concurrent.Immutable;
import javax.annotation.concurrent.NotThreadSafe;
import java.time.Duration;

import static java.util.Objects.requireNonNull;

@Immutable
public final class TuskHeartbeatConfig {

  private final String appName;
  private final Table table;
  private final Duration heartbeatPeriod;
  private final Duration timeout;
  private final BatchingConfig batching;
  private final String appColumn;
  private final String taskColumn;
  private final String sequenceColumn;
  private final String expiryColumn;

  private TuskHeartbeatConfig(
      String appName,
      Table table,
      Duration heartbeatPeriod,
      Duration timeout,
      BatchingConfig batching,
      String appColumn,
      String taskColumn,
      String sequenceColumn,
      String expiryColumn) {
    this.appName = appName;
    this.table = table;
    this.heartbeatPeriod = heartbeatPeriod;
    this.timeout = timeout;
    this.batching = batching;
    this.appColumn = appColumn;
    this.taskColumn = taskColumn;
    this.sequenceColumn = sequenceColumn;
    this.expiryColumn = expiryColumn;
  }

  public static Builder newBuilder() {
    return new Builder();
  }

  public String getAppName() {
    return appName;
  }

  public Table getTable() {
    return table;
  }

  public Duration getHeartbeatPeriod() {
    return heartbeatPeriod;
  }

  public Duration getTimeout() {
    return timeout;
  }

  public BatchingConfig getBatching() {
    return batching;
  }

  public String getAppColumn() {
    return appColumn;
  }

  public String getTaskColumn() {
    return taskColumn;
  }

  public String getSequenceColumn() {
    return sequenceColumn;
  }

  public String getExpiryColumn() {
    return expiryColumn;
  }

  @NotThreadSafe
  public static final class Builder {

    private String appName;
    private Table table;
    private Duration heartbeatPeriod = Duration.ofSeconds(30);
    private Duration timeout = Duration.ofSeconds(30);
    private BatchingConfig batching = BatchingConfig.forDuration(Duration.ofSeconds(5));
    private String appColumn = "app";
    private String taskColumn = "task";
    private String sequenceColumn = "sequence";
    private String expiryColumn = "expiry";

    public Builder setAppName(String appName) {
      this.appName = appName;
      return this;
    }

    public Builder setTable(Table.Builder table) {
      return setTable(table.build());
    }

    public Builder setTable(Table table) {
      this.table = table;
      return this;
    }

    public Builder setHeartbeatPeriod(Duration heartbeatPeriod) {
      this.heartbeatPeriod = heartbeatPeriod;
      return this;
    }

    public Builder setTimeout(Duration timeout) {
      this.timeout = timeout;
      return this;
    }

    public Builder setBatching(BatchingConfig batching) {
      this.batching = batching;
      return this;
    }

    public Builder setAppColumn(String appColumn) {
      this.appColumn = appColumn;
      return this;
    }

    public Builder setTaskColumn(String taskColumn) {
      this.taskColumn = taskColumn;
      return this;
    }

    public Builder setSequenceColumn(String sequenceColumn) {
      this.sequenceColumn = sequenceColumn;
      return this;
    }

    public Builder setExpiryColumn(String expiryColumn) {
      this.expiryColumn = expiryColumn;
      return this;
    }

    public TuskHeartbeatConfig build() {
      requireNonNull(table);
      requireNonNull(heartbeatPeriod);
      requireNonNull(timeout);
      requireNonNull(batching);
      requireNonNull(appColumn);
      requireNonNull(taskColumn);
      requireNonNull(sequenceColumn);
      requireNonNull(expiryColumn);

      BadHeartbeatPeriodException.checkThrow(heartbeatPeriod);
      BadTimeoutException.checkThrow(timeout);

      if (appName == null) {
        appName = TuskUtils.getAppName(appColumn);
      }

      return new TuskHeartbeatConfig(
          appName,
          table,
          heartbeatPeriod,
          timeout,
          batching,
          appColumn,
          taskColumn,
          sequenceColumn,
          expiryColumn);
    }

    private Builder() {}
  }
}
