package io.contek.tusk.heartbeat;

import com.google.common.collect.ConcurrentHashMultiset;
import com.google.common.collect.Multiset;
import io.contek.tusk.Metric;
import io.contek.tusk.Tusk;

import javax.annotation.concurrent.ThreadSafe;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;

import static java.util.concurrent.Executors.newSingleThreadScheduledExecutor;
import static java.util.concurrent.TimeUnit.SECONDS;

@ThreadSafe
public final class TuskHeartbeatClient {

  static final String MAIN_TASK = "main";

  private static final Logger LOGGER = Logger.getLogger(TuskHeartbeatClient.class.getName());

  private final Metric metric;
  private final TuskHeartbeatConfig config;

  private final AtomicBoolean started = new AtomicBoolean(false);
  private final Multiset<String> counts = ConcurrentHashMultiset.create();
  private final ScheduledExecutorService scheduler = newSingleThreadScheduledExecutor();

  private TuskHeartbeatClient(TuskHeartbeatConfig config) {
    this.metric = Metric.metric(config.getTable(), config.getBatching());
    this.config = config;
  }

  public static TuskHeartbeatClient forConfig(TuskHeartbeatConfig config) {
    return new TuskHeartbeatClient(config);
  }

  public void start() {
    if (started.getAndSet(true)) {
      throw new IllegalStateException("Already started");
    }

    scheduler.scheduleAtFixedRate(
        this::sendMainBeat, 0, config.getHeartbeatPeriod().getSeconds(), SECONDS);
  }

  public void beat(String task, Duration heartbeatPeriod) {
    if (!started.get()) {
      throw new IllegalStateException("Client not started");
    }

    BadTaskNameException.checkThrow(task);
    sendBeat(task, heartbeatPeriod);
  }

  private void sendMainBeat() {
    try {
      sendBeat(MAIN_TASK, config.getHeartbeatPeriod());
    } catch (Throwable t) {
      LOGGER.log(Level.SEVERE, "Unknown Error.", t);
    }
  }

  private void sendBeat(String task, Duration heartbeatPeriod) {
    Clock clock = Tusk.getClock();
    Instant expiry =
        clock
            .instant()
            .plus(heartbeatPeriod)
            .plus(config.getTimeout())
            .plus(config.getBatching().getPeriod());
    metric
        .newEntry()
        .putString(config.getAppColumn(), config.getAppName())
        .putString(config.getTaskColumn(), task)
        .putUInt32(config.getSequenceColumn(), counts.add(task, 1))
        .putDateTime64(config.getExpiryColumn(), expiry)
        .write();
  }
}
