package io.contek.tusk;

import com.clickhouse.client.*;
import com.clickhouse.data.ClickHouseDataType;
import com.clickhouse.data.ClickHouseRecord;

import javax.annotation.Nullable;
import javax.annotation.concurrent.NotThreadSafe;
import javax.annotation.concurrent.ThreadSafe;
import java.time.Duration;
import java.util.function.Consumer;
import java.util.logging.Logger;

import static com.clickhouse.client.ClickHouseCredentials.fromUserAndPassword;
import static com.clickhouse.client.ClickHouseProtocol.HTTP;
import static com.clickhouse.client.config.ClickHouseClientOption.*;
import static com.clickhouse.data.ClickHouseCompression.ZSTD;
import static com.clickhouse.data.ClickHouseFormat.TabSeparatedWithNamesAndTypes;
import static java.lang.String.format;
import static java.util.Objects.requireNonNull;
import static java.util.logging.Level.SEVERE;
import static java.util.logging.Logger.getLogger;

@ThreadSafe
public final class MetricClient {

  private static final Logger LOGGER = getLogger(MetricClient.class.getName());

  private final ClickHouseClient client;
  private final ClickHouseNode server;

  private MetricClient(ClickHouseClient client, ClickHouseNode server) {
    this.client = client;
    this.server = server;
  }

  public static Builder newBuilder() {
    return new Builder();
  }

  @Nullable
  Schema describe(Table table) {
    try {
      String query = format("DESCRIBE TABLE %s", table.getName());
      ClickHouseResponse response =
          newRequest(table.getDatabase())
              .format(TabSeparatedWithNamesAndTypes)
              .query(query)
              .execute()
              .get();
      Iterable<ClickHouseRecord> records = response.records();

      Schema.Builder result = Schema.newBuilder();
      for (ClickHouseRecord record : records) {
        String columnName = record.getValue("name").asString();
        String columnType = record.getValue("type").asString();
        ClickHouseDataType type = parseType(columnType);
        result.add(columnName, type);
      }

      return result.build();
    } catch (Throwable t) {
      LOGGER.log(SEVERE, format("Failed to describe table %s.", table), t);
      return null;
    }
  }

  void write(MetricData data, Consumer<Throwable> onError) {
    newRequest(data.getTable().getDatabase())
        .write()
        .table(data.getTable().getName())
        .format(data.getFormat())
        .data(data.getInputStream())
        .seal()
        .execute()
        .exceptionally(
            t -> {
              LOGGER.log(SEVERE, format("Failed to write data into table %s.", data.getTable()), t);
              onError.accept(t);
              return null;
            });
  }

  private ClickHouseRequest<?> newRequest(@Nullable String database) {
    ClickHouseRequest<?> request = client.read(server);
    if (database != null) {
      request.use(database);
    }
    return request;
  }

  private static ClickHouseDataType parseType(String typeName) {
    if (!typeName.endsWith(")")) {
      return ClickHouseDataType.of(typeName);
    }

    int bracketIndex = typeName.indexOf('(');
    String typeNameOrModifier = typeName.substring(0, bracketIndex);
    return switch (typeNameOrModifier) {
      case "LowCardinality", "Nullable" -> ClickHouseDataType.of(
          typeName.substring(bracketIndex + 1, typeName.length() - 1));
      default -> ClickHouseDataType.of(typeNameOrModifier);
    };
  }

  @NotThreadSafe
  public static final class Builder {

    private String host = "localhost";
    private String user = "default";
    private Duration connectTimeout = Duration.ofSeconds(60);
    private Duration socketTimeout = Duration.ofSeconds(180);

    private String password;
    private String database;
    private Integer port = 8123;

    private boolean secure=false;
    public Builder setHost(String host) {
      this.host = host;
      return this;
    }
    public Builder setPort(Integer port) {
      this.port = port;
      return this;
    }
    public Builder setSecure(Boolean secure) {
      this.secure = secure;
      return this;
    }

    public Builder setUser(String user) {
      this.user = user;
      return this;
    }

    public Builder setConnectTimeout(Duration connectTimeout) {
      this.connectTimeout = connectTimeout;
      return this;
    }

    public Builder setSocketTimeout(Duration socketTimeout) {
      this.socketTimeout = socketTimeout;
      return this;
    }

    public Builder setPassword(@Nullable String password) {
      this.password = password;
      return this;
    }

    public Builder setDatabase(@Nullable String database) {
      this.database = database;
      return this;
    }

    public MetricClient build() {
      requireNonNull(connectTimeout);

      ClickHouseClient client =
          ClickHouseClient.builder()
              .nodeSelector(ClickHouseNodeSelector.of(HTTP))
              .option(CONNECTION_TIMEOUT, (int) connectTimeout.toMillis())
              .option(SOCKET_TIMEOUT, (int) socketTimeout.toMillis())
              .option(COMPRESS_ALGORITHM, ZSTD)
              .build();
      ClickHouseNode server = createConnection();
      return new MetricClient(client, server);
    }

    private ClickHouseNode createConnection() {
      requireNonNull(host);
      requireNonNull(user);

      return ClickHouseNode.builder()
          .host(host)
          .database(database)
          .credentials(fromUserAndPassword(user, password))
          .port(HTTP, port).addOption("ssl", secure?"true":"false")
          .build();
    }

    private Builder() {}
  }
}
