package io.contek.tusk;

import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;
import java.time.Duration;

import static java.lang.String.format;

@Immutable
public final class BatchingConfig {

  private static final Duration DEFAULT_BATCHING = Duration.ofSeconds(60);
  private static final int DEFAULT_BATCH_SIZE = 10000;

  private final Duration period;
  private final Integer maxSize;

  private BatchingConfig(Duration period, @Nullable Integer maxSize) {
    this.period = period;
    this.maxSize = maxSize;
  }

  public static BatchingConfig disabled() {
    return new BatchingConfig(Duration.ZERO, null);
  }

  public static BatchingConfig getDefault() {
    return forDuration(DEFAULT_BATCHING, DEFAULT_BATCH_SIZE);
  }

  public static BatchingConfig forDuration(Duration period) {
    return forDuration(period, null);
  }

  public static BatchingConfig forDuration(Duration period, @Nullable Integer maxSize) {
    if (period.isNegative() || period.isZero()) {
      throw new IllegalArgumentException(format("Illegal batching period %s", period));
    }
    if (maxSize != null && maxSize <= 0) {
      throw new IllegalArgumentException(format("Illegal max batch size %s", maxSize));
    }
    return new BatchingConfig(period, maxSize);
  }

  public Duration getPeriod() {
    return period;
  }

  @Nullable
  public Integer getMaxSize() {
    return maxSize;
  }
}
