package io.contek.tusk;

import com.clickhouse.data.ClickHouseDataType;
import com.clickhouse.data.format.BinaryStreamUtils;

import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.math.BigInteger;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.List;
import java.util.TimeZone;
import java.util.function.BiConsumer;

import static com.clickhouse.data.ClickHouseFormat.RowBinary;

@Immutable
final class MetricFormatter {

  private final Table table;
  private final SchemaProvider provider;

  MetricFormatter(Table table, SchemaProvider provider) {
    this.table = table;
    this.provider = provider;
  }

  @Nullable
  public MetricData format(List<MetricRow> rows) throws MetricFormatException {
    Schema schema = provider.getSchema();
    if (schema == null || rows.isEmpty()) {
      return null;
    }

    try (ByteArrayOutputStream output = new ByteArrayOutputStream()) {
      for (MetricRow row : rows) {
        schema.getColumns().forEach(findAndWriteValue(row, output));
      }
      return new MetricData(table, output.toByteArray(), RowBinary);
    } catch (IOException e) {
      throw new MetricFormatException(e);
    }
  }

  private static BiConsumer<String, ClickHouseDataType> findAndWriteValue(
      MetricRow row, OutputStream output) {
    return (column, type) -> {
      Object value = row.getValue(column);
      writeValue(value, type, output);
    };
  }

  private static void writeValue(
      @Nullable Object value, ClickHouseDataType type, OutputStream output) {
    try {
      switch (type) {
        case DateTime -> {
          Instant instant = value == null ? Instant.EPOCH : (Instant) value;
          ZoneId tz = Tusk.getClock().getZone();
          BinaryStreamUtils.writeDateTime(
              output, LocalDateTime.ofInstant(instant, tz), TimeZone.getTimeZone(tz));
        }
        case DateTime64 -> {
          Instant instant = value == null ? Instant.EPOCH : (Instant) value;
          ZoneId tz = Tusk.getClock().getZone();
          BinaryStreamUtils.writeDateTime64(
              output, LocalDateTime.ofInstant(instant, tz), TimeZone.getTimeZone(tz));
        }
        case Bool -> {
          boolean bool = value != null && (boolean) value;
          BinaryStreamUtils.writeBoolean(output, bool);
        }
        case String -> {
          String str = value == null ? "" : (String) value;
          BinaryStreamUtils.writeString(output, str);
        }
        case UInt8 -> {
          int uInt8 = value == null ? 0 : (int) value;
          BinaryStreamUtils.writeUnsignedInt8(output, uInt8);
        }
        case UInt16 -> {
          int uInt16 = value == null ? 0 : (int) value;
          BinaryStreamUtils.writeUnsignedInt16(output, uInt16);
        }
        case UInt32 -> {
          long uInt32 = value == null ? 0L : (long) value;
          BinaryStreamUtils.writeUnsignedInt32(output, uInt32);
        }
        case UInt64 -> {
          BigInteger uInt64 = value == null ? BigInteger.ZERO : (BigInteger) value;
          BinaryStreamUtils.writeUnsignedInt64(output, uInt64);
        }
        case Int8 -> {
          byte int8 = value == null ? 0 : (byte) value;
          BinaryStreamUtils.writeInt8(output, int8);
        }
        case Int16 -> {
          short int16 = value == null ? 0 : (short) value;
          BinaryStreamUtils.writeInt16(output, int16);
        }
        case Int32 -> {
          int int32 = value == null ? 0 : (int) value;
          BinaryStreamUtils.writeInt32(output, int32);
        }
        case Int64 -> {
          long int64 = value == null ? 0L : (long) value;
          BinaryStreamUtils.writeInt64(output, int64);
        }
        case Float32 -> {
          float float32 = value == null ? 0F : (float) value;
          BinaryStreamUtils.writeFloat32(output, float32);
        }
        case Float64 -> {
          double float64 = value == null ? 0D : (double) value;
          BinaryStreamUtils.writeFloat64(output, float64);
        }
        default -> throw new UnsupportedOperationException(type.name());
      }
    } catch (Throwable e) {
      throw new MetricFormatException(e);
    }
  }
}
